#!/usr/bin/env python3
"""Maintain metadata for standalone manuscript HTML files.

The metadata captures lightweight analytics (word counts, heading totals),
source provenance tags, and review notes so the admin portal can surface
change tracking and internal review status.
"""
from __future__ import annotations

import argparse
import json
import sys
from pathlib import Path
from typing import List

ROOT = Path(__file__).resolve().parents[1]
sys.path.insert(0, str(ROOT / "omega_document" / "admin_portal"))

from metadata_utils import REVIEW_STATES, update_metadata, load_metadata  # noqa: E402


def parse_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser(description=__doc__)
    parser.add_argument("slug", help="Standalone document slug (e.g. rft-volume1-final)")
    parser.add_argument(
        "--source",
        action="append",
        dest="sources",
        help="Append a source/provenance tag (can be passed multiple times).",
    )
    parser.add_argument(
        "--set-sources",
        help="Comma-separated list of sources to replace the current set.",
    )
    parser.add_argument(
        "--clear-sources",
        action="store_true",
        help="Remove all recorded sources.",
    )
    parser.add_argument(
        "--review-status",
        choices=list(REVIEW_STATES),
        help="Update the review status field.",
    )
    parser.add_argument(
        "--note",
        help="Append a review note/comment (timestamped).",
    )
    parser.add_argument(
        "--print",
        action="store_true",
        help="Print the resulting metadata to stdout after updates.",
    )
    return parser.parse_args()


def main() -> int:
    args = parse_args()
    slug = args.slug
    append_sources: List[str] = args.sources or []
    set_sources = (
        [s.strip() for s in args.set_sources.split(",") if s.strip()]
        if args.set_sources is not None
        else None
    )

    metadata = update_metadata(
        slug,
        review_status=args.review_status,
        note=args.note,
        set_sources=set_sources,
        append_sources=append_sources,
        clear_sources=args.clear_sources,
        recompute_stats=True,
    )

    if args.print:
        print(json.dumps(metadata.data, indent=2, ensure_ascii=False))

    return 0


if __name__ == "__main__":
    sys.exit(main())
