#!/usr/bin/env python3
"""Ensure the public site lists the RC11 preprint in papers.json."""

from __future__ import annotations

import argparse
import json
from datetime import date
from pathlib import Path
from typing import Any, Dict, List


ROOT = Path(__file__).resolve().parents[1]
PAPERS_JSON = ROOT / "rft-cosmology-site" / "app" / "static" / "data" / "papers.json"


DEFAULT_ENTRY: Dict[str, Any] = {
    "id": "volume1-rc11",
    "title": "RFT Volume 1: Unified Field Dynamics (RC11)",
    "series": "Volume1",
    "status": "public",
    "pubDate": "2025-09-21",
    "doi": None,
    "abstract": "Accessibility-focused RC11 build of Volume 1 with preregistered gates and instant verification hooks.",
    "filename": "RFT_Vol1_v1.0_RC11_PREPRINT.pdf"
}


def load_entries() -> List[Dict[str, Any]]:
    if not PAPERS_JSON.exists():
        raise FileNotFoundError(f"papers.json not found at {PAPERS_JSON}")
    return json.loads(PAPERS_JSON.read_text(encoding="utf-8"))


def save_entries(entries: List[Dict[str, Any]]) -> None:
    PAPERS_JSON.write_text(json.dumps(entries, indent=2) + "\n", encoding="utf-8")


def ensure_entry(entries: List[Dict[str, Any]], template: Dict[str, Any]) -> List[Dict[str, Any]]:
    updated = False
    for entry in entries:
        if entry.get("filename") == template["filename"] or entry.get("id") == template["id"]:
            entry.update(template)
            updated = True
            break
    if not updated:
        entries.append(template)

    def sort_key(item: Dict[str, Any]) -> date:
        try:
            year, month, day = map(int, item.get("pubDate", "1970-01-01").split("-"))
            return date(year, month, day)
        except Exception:
            return date(1970, 1, 1)

    entries.sort(key=sort_key, reverse=True)
    return entries


def main() -> int:
    parser = argparse.ArgumentParser(description=__doc__)
    parser.add_argument(
        "--title",
        default=DEFAULT_ENTRY["title"],
        help="Override title for the RC11 entry",
    )
    parser.add_argument(
        "--pubdate",
        default=DEFAULT_ENTRY["pubDate"],
        help="Publication date (YYYY-MM-DD)",
    )
    parser.add_argument(
        "--abstract",
        default=DEFAULT_ENTRY["abstract"],
        help="Short abstract/description",
    )
    args = parser.parse_args()

    template = DEFAULT_ENTRY.copy()
    template["title"] = args.title
    template["pubDate"] = args.pubdate
    template["abstract"] = args.abstract

    entries = ensure_entry(load_entries(), template)
    save_entries(entries)
    print(f"Ensured RC11 entry present in {PAPERS_JSON}")
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
