#!/usr/bin/env node

const fs = require('fs');
const path = require('path');
const Fuse = require('fuse.js');

// Performance test for search functionality
class SearchPerformanceTester {
    constructor() {
        this.indexPath = path.join(__dirname, '..', 'app', 'static', 'search', 'index.json');
        this.testQueries = [
            // Single word queries
            'scalaron',
            'twistor',
            'neutrino',
            'inflation',
            'echo',
            
            // Multi-word queries
            'scalaron mass',
            'gauge coupling',
            'dark matter',
            
            // Equation searches
            'Δt_echo',
            'Λ_GUT',
            'M^4',
            
            // Parameter searches
            '1.6e14',
            'param:scalaron_mass_GeV',
            
            // Complex queries
            'twistor path integral',
            'Einstein inflation'
        ];
    }

    loadIndex() {
        try {
            const indexData = fs.readFileSync(this.indexPath, 'utf8');
            this.index = JSON.parse(indexData);
            console.log(`✓ Loaded search index with ${this.index.length} entries`);
        } catch (error) {
            console.error('✗ Failed to load search index:', error.message);
            process.exit(1);
        }
    }

    initializeFuse() {
        const fuseOptions = {
            keys: [
                { name: 'title', weight: 0.4 },
                { name: 'text', weight: 0.3 },
                { name: 'latex', weight: 0.2 },
                { name: 'context', weight: 0.1 }
            ],
            threshold: 0.3,
            includeScore: true,
            includeMatches: true,
            minMatchCharLength: 2,
            useExtendedSearch: true
        };
        
        this.fuse = new Fuse(this.index, fuseOptions);
        console.log('✓ Initialized Fuse.js search engine');
    }

    runPerformanceTests() {
        console.log('\n🔍 Running search performance tests...\n');
        
        const results = [];
        let totalTime = 0;
        let slowQueries = [];
        
        this.testQueries.forEach(query => {
            // Warm up run
            this.fuse.search(query);
            
            // Measure 10 runs
            const times = [];
            for (let i = 0; i < 10; i++) {
                const start = performance.now();
                const searchResults = this.fuse.search(query);
                const end = performance.now();
                times.push(end - start);
            }
            
            // Calculate average
            const avgTime = times.reduce((a, b) => a + b, 0) / times.length;
            const minTime = Math.min(...times);
            const maxTime = Math.max(...times);
            
            const result = {
                query,
                avgTime: avgTime.toFixed(2),
                minTime: minTime.toFixed(2),
                maxTime: maxTime.toFixed(2),
                resultCount: this.fuse.search(query).length
            };
            
            results.push(result);
            totalTime += avgTime;
            
            // Track slow queries
            if (avgTime > 50) {
                slowQueries.push(result);
            }
            
            // Print result
            const status = avgTime < 50 ? '✓' : '✗';
            const color = avgTime < 50 ? '\x1b[32m' : '\x1b[31m';
            console.log(`${color}${status}\x1b[0m "${query}" - Avg: ${avgTime.toFixed(2)}ms (${result.resultCount} results)`);
        });
        
        // Summary
        console.log('\n📊 Performance Summary:');
        console.log('─'.repeat(50));
        console.log(`Total queries tested: ${this.testQueries.length}`);
        console.log(`Average query time: ${(totalTime / this.testQueries.length).toFixed(2)}ms`);
        console.log(`Queries under 50ms: ${this.testQueries.length - slowQueries.length}/${this.testQueries.length}`);
        
        if (slowQueries.length > 0) {
            console.log('\n⚠️  Slow queries (>50ms):');
            slowQueries.forEach(q => {
                console.log(`  - "${q.query}": ${q.avgTime}ms`);
            });
        }
        
        // Performance benchmark
        const avgQueryTime = totalTime / this.testQueries.length;
        if (avgQueryTime < 50) {
            console.log('\n✅ PASS: Average query time is under 50ms requirement');
        } else {
            console.log('\n❌ FAIL: Average query time exceeds 50ms requirement');
            process.exit(1);
        }
        
        // Save detailed results
        const reportPath = path.join(__dirname, '..', 'search-performance-report.json');
        fs.writeFileSync(reportPath, JSON.stringify({
            timestamp: new Date().toISOString(),
            indexSize: this.index.length,
            results,
            summary: {
                totalQueries: this.testQueries.length,
                averageTime: (totalTime / this.testQueries.length).toFixed(2),
                slowQueries: slowQueries.length,
                passed: avgQueryTime < 50
            }
        }, null, 2));
        
        console.log(`\n📄 Detailed report saved to: ${reportPath}`);
    }

    run() {
        console.log('🚀 RFT Search Performance Test Suite\n');
        
        this.loadIndex();
        this.initializeFuse();
        this.runPerformanceTests();
    }
}

// Check dependencies
try {
    require('fuse.js');
} catch (e) {
    console.error('Missing dependencies. Please run: npm install');
    process.exit(1);
}

// Run tests
if (require.main === module) {
    const tester = new SearchPerformanceTester();
    tester.run();
}