#!/usr/bin/env python3
"""
Selftest script for SPARC-120 pipeline verification.
No pytest required - runs CLI directly and validates results.
"""
import json
import sys
import subprocess
import shlex
import hashlib
import time
from pathlib import Path

ROOT = Path(__file__).resolve().parents[1]
PY = sys.executable

def run_cli_case(case_path: Path, extra=None):
    """Run a single case through the CLI bench runner."""
    extra = extra or []
    cmd = [
        PY, "-m", "cli.rft_rc_bench",
        "--case", str(case_path),
        "--solver", "rft_geom",
        "--global-config", "config/global.json",
        "--residuals", "--dump-modes"
    ] + extra
    print("\n>>", " ".join(shlex.quote(c) for c in cmd))
    r = subprocess.run(cmd, cwd=ROOT, text=True, capture_output=True)
    print(r.stdout)
    if r.returncode != 0:
        print(r.stderr, file=sys.stderr)
        raise SystemExit(r.returncode)

    # Read metrics - use galaxy name from case file, not filename
    case_data = json.loads(case_path.read_text())
    galaxy_slug = case_data["name"].replace(" ", "_")
    mpath = ROOT / "reports" / galaxy_slug / "metrics.json"
    with mpath.open() as f:
        return json.load(f)

def aggregate(suffix=None):
    """Run the aggregator to generate summary."""
    cmd = [PY, "-m", "batch.aggregate"]
    if suffix:
        cmd.extend(["--suffix", suffix])
    print("\n>>", " ".join(cmd))
    r = subprocess.run(cmd, cwd=ROOT, text=True, capture_output=True)
    print(r.stdout)
    if r.returncode != 0:
        print(r.stderr, file=sys.stderr)
        raise SystemExit(r.returncode)

def verify_config_hash():
    """Verify config hash matches golden guard."""
    config_path = ROOT / "config" / "global.json"
    expected = "dd1b2e29d27ee42d5a2601b588f1a82d2a20343d1fdbc0842fd5683d9de64218"
    digest = hashlib.sha256(config_path.read_bytes()).hexdigest()
    if digest != expected:
        print(f"⚠️  Config hash mismatch!")
        print(f"   Expected: {expected}")
        print(f"   Got:      {digest}")
        print("   If config change was intentional, update test_golden_no_retune.py")
        return False
    return True

def main():
    print("=" * 70)
    print("SPARC-120 Pipeline Selftest")
    print("=" * 70)

    # 1. Verify config hash
    print("\n[1/4] Verifying config hash guard...")
    if not verify_config_hash():
        print("❌ Config hash check FAILED")
        return 2
    print("✅ Config hash OK")

    # 2. Run golden cases
    print("\n[2/4] Running golden cases (MW + NGC 2403)...")

    # Load config and expectations
    with (ROOT / "config" / "global.json").open() as f:
        config = json.load(f)

    golden_cases = config["no_retune_guard"]["golden_cases"]
    expectations = config["no_retune_guard"]["expectations"]

    passed = 0
    for galaxy_key in golden_cases:
        # Convert config key to filename (e.g., "MilkyWay" -> "MilkyWay.json")
        case_path = ROOT / "cases" / f"{galaxy_key}.json"
        if not case_path.exists():
            print(f"❌ Missing case: {case_path}")
            return 3

        result = run_cli_case(case_path)
        galaxy_name = result["galaxy"]
        m = result["metrics"]
        exp = expectations[galaxy_key]

        # Check pass status
        case_pass = m.get("pass", False)

        # Check against expectations
        rms_ok = m.get("rms_percent", 999) <= exp.get("rms_pct_max", 0)
        mae_ok = m.get("mae_percent", 999) <= exp.get("mae_pct_max", 0)

        status = "✅" if (case_pass and rms_ok and mae_ok) else "❌"
        print(f"{status} {galaxy_name}: PASS={case_pass}  RMS%={m.get('rms_percent'):.2f}  MAE%={m.get('mae_percent'):.2f}")

        if case_pass and rms_ok and mae_ok:
            passed += 1
        else:
            print(f"   Expected: RMS% ≤ {exp.get('rms_pct_max')}, MAE% ≤ {exp.get('mae_pct_max')}")

    if passed != len(golden_cases):
        print("\n❌ Selftest FAIL: not all golden cases passed")
        return 4

    # 3. Test aggregator
    print("\n[3/4] Testing aggregator...")
    aggregate()
    summary_path = ROOT / "reports" / "_summary.json"
    if not summary_path.exists():
        print("❌ Aggregator did not create _summary.json")
        return 5

    with summary_path.open() as f:
        summary = json.load(f)
    print(f"✅ Summary generated: {summary.get('n_cases', 0)} cases")

    # Test with suffix
    aggregate(suffix="TEST")
    suffix_path = ROOT / "reports" / "_summary_TEST.json"
    if not suffix_path.exists():
        print("❌ Aggregator did not create _summary_TEST.json")
        return 6
    print("✅ Suffix support OK")

    # 4. Verify dashboard files
    print("\n[4/4] Checking dashboard files...")
    dashboard_files = [
        ROOT / "web" / "index.html",
        ROOT / "web" / "app.js"
    ]
    for f in dashboard_files:
        if not f.exists():
            print(f"❌ Missing dashboard file: {f}")
            return 7
    print("✅ Dashboard files present")

    print("\n" + "=" * 70)
    print("✅ Selftest PASSED")
    print("=" * 70)
    print("\nNext steps:")
    print("  1. Get SPARC data and run: python -m ingest.make_manifest --root /path/to/SPARC/rotmod")
    print("  2. Select cohort: python -m ingest.select_cohort --summary reports/_summary.json --n 120")
    print("  3. View dashboard: python3 -m http.server -d web 8080")

    return 0

if __name__ == "__main__":
    sys.exit(main())
