#!/bin/bash
# V2 Calibrated Grid: 12 configs addressing amplitude & gate suppression
# A0 ∈ {400, 1000, 1800}, α ∈ {0.6, 0.8}, g* ∈ {1000, 2500}
# γ = 0.5 (softer gate), r_turn = 2.0, p = 2.0

set -e

TRAIN="cases/SP99-TRAIN.manifest.txt"
RESULTS_DIR="results/v2_calibrated"
mkdir -p "$RESULTS_DIR"

echo "========================================================================"
echo "RFT V2 CALIBRATED GRID: 12 Configs"
echo "========================================================================"
echo "Addressing root causes:"
echo "  1. Amplitude too low (A0: 400-1800 vs prior 200-260)"
echo "  2. Gate over-suppression (γ=0.5 vs prior 1.0-1.5)"
echo ""
echo "Gate: 🟢 GO if pass@20% ≥30% on TRAIN"
echo "      🟡 STOP if 20-30%"
echo "      🔴 STOP if <20%"
echo "========================================================================"
echo ""

config_id=0
best_pass_rate=0
best_config=""

for A0 in 400 1000 1800; do
  for ALPHA in 0.6 0.8; do
    for GSTAR in 1000 2500; do
      echo "------------------------------------------------------------------------"
      echo "[Config $config_id/12] A0=$A0, α=$ALPHA, g*=$GSTAR, γ=0.5"
      echo "------------------------------------------------------------------------"

      # Run with this config
      python3 << EOF
import sys
from pathlib import Path
sys.path.insert(0, str(Path.cwd()))

import json
import numpy as np
from sparc_rft.case import load_case
from solver.rft_v2_gated_tail import apply_v2_gated_tail
from metrics.rc_metrics import compute_metrics

kernel_config = {
    "grid": [0.0],
    "weights": [1.0],
    "r_scale": "r_geo"
}

tail_config = {
    "A0_kms2_per_kpc": $A0,
    "alpha": $ALPHA,
    "g_star_kms2_per_kpc": $GSTAR,
    "gamma": 0.5,  # Soft gate
    "r_turn_kpc": 2.0,
    "p": 2.0
}

# Load manifest
with open("$TRAIN") as f:
    case_paths = [f"cases/{line.strip()}" for line in f if line.strip() and not line.startswith("#")]

results = []
for case_path in case_paths:
    try:
        case = load_case(case_path)
        result = apply_v2_gated_tail(case, kernel_config, tail_config)

        r = np.array(result["r_kpc"])
        v_pred = np.array(result["v_pred_kms"])
        v_obs = np.array(case.v_obs_kms)
        sigma = np.array(case.sigma_v_kms)

        mask = (r >= 1.0) & (r <= 30.0) & (sigma > 0)
        if mask.sum() < 3:
            continue

        metrics = compute_metrics(v_obs_kms=v_obs[mask], v_pred_kms=v_pred[mask])

        results.append({
            "name": case.name,
            "rms_percent": metrics["rms_percent"],
            "pass_20": metrics["rms_percent"] <= 20.0
        })
    except Exception as e:
        print(f"  ERROR {case_path}: {e}", file=sys.stderr)

# Aggregate
if results:
    pass_20_count = sum(r["pass_20"] for r in results)
    pass_20_rate = 100.0 * pass_20_count / len(results)
    rms_median = np.median([r["rms_percent"] for r in results])

    summary = {
        "config_id": $config_id,
        "A0": $A0,
        "alpha": $ALPHA,
        "g_star": $GSTAR,
        "gamma": 0.5,
        "n_galaxies": len(results),
        "pass_20_count": int(pass_20_count),
        "pass_20_rate": float(pass_20_rate),
        "rms_percent_median": float(rms_median),
        "results": results
    }

    with open("$RESULTS_DIR/config_${config_id}.json", "w") as f:
        json.dump(summary, f, indent=2)

    print(f"  ✓ {len(results)} galaxies")
    print(f"  Pass@20%: {pass_20_count}/{len(results)} = {pass_20_rate:.1f}%")
    print(f"  RMS median: {rms_median:.1f}%")
    print("")
else:
    print("  ✗ No results")
    print("")
EOF

      # Track best
      pass_rate=$(jq -r '.pass_20_rate // 0' "$RESULTS_DIR/config_${config_id}.json")
      if (( $(echo "$pass_rate > $best_pass_rate" | bc -l) )); then
        best_pass_rate=$pass_rate
        best_config=$config_id
      fi

      config_id=$((config_id + 1))
    done
  done
done

echo "========================================================================"
echo "GRID COMPLETE"
echo "========================================================================"
echo "Best config: $best_config with ${best_pass_rate}% pass@20%"
echo ""

# Gate decision
if (( $(echo "$best_pass_rate >= 30" | bc -l) )); then
  echo "🟢 GREEN GATE: ${best_pass_rate}% ≥ 30%"
  echo "   → Proceed to TEST validation"
elif (( $(echo "$best_pass_rate >= 20" | bc -l) )); then
  echo "🟡 YELLOW GATE: ${best_pass_rate}% ∈ [20%, 30%)"
  echo "   → Document and stop"
else
  echo "🔴 RED GATE: ${best_pass_rate}% < 20%"
  echo "   → Close v2 track"
fi

echo ""
echo "Results saved to: $RESULTS_DIR/"
echo "========================================================================"
