#!/bin/bash
# C9 TEST Validation - Run after grid search completes
# This script executes the blind validation on SP99-TEST

set -e

TRAIN_SUMMARY="reports/_summary_C9_A.json"
TEST_MANIFEST="cases/SP99-TEST.manifest.txt"
CONFIG="config/global_c9.json"

echo "=== C9 TEST Validation Sequence ==="
echo ""

# Check prerequisites
if [ ! -f "$CONFIG" ]; then
    echo "ERROR: Config not found: $CONFIG"
    echo "Run scripts/extract_c9_winner.py first to freeze optimal config"
    exit 1
fi

if [ ! -f "$TEST_MANIFEST" ]; then
    echo "ERROR: TEST manifest not found: $TEST_MANIFEST"
    exit 1
fi

# Step 1: Run RFT solver on TEST
echo "Step 1/5: Running RFT solver on TEST set (34 galaxies)..."
python3 -m cli.rft_rc_bench \
    --batch "$TEST_MANIFEST" \
    --solver rft_geom \
    --global-config "$CONFIG" \
    --min-radius 1.0 \
    --max-radius 30.0 \
    --min-points 10 \
    --emit-curves \
    --max-workers 0

# Step 2: Run baseline solvers on TEST
echo ""
echo "Step 2/5: Running baseline solvers (MOND + NFW) on TEST..."

python3 -m cli.rft_rc_bench \
    --batch "$TEST_MANIFEST" \
    --solver mond \
    --global-config config/global.json \
    --min-radius 1.0 \
    --max-radius 30.0 \
    --min-points 10 \
    --emit-curves \
    --max-workers 0

python3 -m cli.rft_rc_bench \
    --batch "$TEST_MANIFEST" \
    --solver nfw_fit \
    --global-config config/global.json \
    --nfw-bounds "rho_s=1e6,1e9;r_s=0.1,100" \
    --min-radius 1.0 \
    --max-radius 30.0 \
    --min-points 10 \
    --emit-curves \
    --max-workers 0

# Step 3: Aggregate with manifest restriction
echo ""
echo "Step 3/5: Aggregating TEST results with manifest restriction..."
python3 -m batch.aggregate \
    --restrict-manifest "$TEST_MANIFEST" \
    --pass-threshold 20.0 \
    --suffix C9_TEST

# Step 4: Validate manifest filtering worked
echo ""
echo "Step 4/5: Validating manifest filtering..."
test_count=$(jq '.rows | length' reports/_summary_C9_TEST.json)
manifest_count=$(grep -v '^#' "$TEST_MANIFEST" | wc -l)

if [ "$test_count" -ne "$manifest_count" ]; then
    echo "WARNING: Row count ($test_count) != manifest count ($manifest_count)"
    echo "Manifest filtering may have failed!"
else
    echo "✓ Manifest filtering verified: $test_count galaxies"
fi

# Step 5: Analyze worst performers
echo ""
echo "Step 5/5: Analyzing worst 10 performers..."
./scripts/analyze_worst10.sh reports/_summary_C9_TEST.json

# Report results
echo ""
echo "=== C9 TEST Results ==="
test_passes=$(jq '[.rows[] | select(.solvers.rft_geom.rms_pct <= 20)] | length' reports/_summary_C9_TEST.json)
test_median=$(jq '.solver_totals.rft_geom.rms_pct_median' reports/_summary_C9_TEST.json)
test_pct=$(awk "BEGIN {printf \"%.1f\", $test_passes*100/34}")
echo "RFT pass@20%: $test_passes/34 (${test_pct}%)"
echo "Median RMS: ${test_median}%"

# BIC wins
rft_mond_bic=$(jq '.bic_wins["rft_geom>mond"].wins' reports/_summary_C9_TEST.json 2>/dev/null || echo "N/A")
rft_nfw_bic=$(jq '.bic_wins["rft_geom>nfw_fit"].wins' reports/_summary_C9_TEST.json 2>/dev/null || echo "N/A")
echo "BIC wins: RFT>MOND=$rft_mond_bic, RFT>NFW=$rft_nfw_bic"

echo ""
echo "✅ TEST validation complete!"
echo "Summary: reports/_summary_C9_TEST.json"
echo "Diagnostics: reports/worst10_analysis/"
