#!/usr/bin/env python3
"""Automation helper to promote an Omega portal build to the public site.

The script copies the standalone HTML/PDF artifacts produced by the Omega
portal into the Flask application's `public/publications/<slug>/`
folder, ensuring every release is mirrored inside the repository for
versioning and future GitHub exports.

Usage
-----
    python scripts/publish_publication.py --slug rft-volume2-rc12 \
        --source-slug rft-volume2-rc12 \
        --dest public/publications

* `--slug` is the name used in the manifest and public URLs.
* `--source-slug` defaults to the same value and corresponds to the
  folder names inside the Omega portal's `static/html/standalone`
  and `static/standalone_assets` directories.
* `--dest` controls where the publication bundle is staged inside this
  repository (defaults to `public/publications`).

The command copies the following assets when they exist:

- HTML builds (`.html`, `.static.html`, `-simple.html`, `-enhanced.html`)
- PDFs (`.pdf`, `.static.pdf`)
- Any extra files referenced in `standalone_assets/<slug>/` are mirrored
  into a sibling `assets/` directory so that figures / notebooks remain
  available.

After running the script, re-run `scripts/generate_publications_manifest.py`
and commit both the manifest and the copied assets.
"""

from __future__ import annotations

import argparse
import shutil
from pathlib import Path

OMEGA_STATIC = Path("/home/rftuser/omega_document/admin_portal/static")
HTML_ROOT = OMEGA_STATIC / "html" / "standalone"
ASSETS_ROOT = OMEGA_STATIC / "standalone_assets"


def copy_if_exists(src: Path, dest: Path) -> None:
    if not src.exists():
        return
    dest.parent.mkdir(parents=True, exist_ok=True)
    if src.is_dir():
        if dest.exists():
            shutil.rmtree(dest)
        shutil.copytree(src, dest)
    else:
        shutil.copy2(src, dest)


def stage_publication(slug: str, source_slug: str, dest_root: Path) -> None:
    dest_root = dest_root / slug
    dest_root.mkdir(parents=True, exist_ok=True)

    # Copy HTML variants
    html_variants = [
        f"{source_slug}.html",
        f"{source_slug}.static.html",
        f"{source_slug}-simple.html",
        f"{source_slug}-enhanced.html",
    ]
    for name in html_variants:
        copy_if_exists(HTML_ROOT / name, dest_root / name)

    # Copy PDFs
    pdf_variants = [
        f"{source_slug}.pdf",
        f"{source_slug}.static.pdf",
    ]
    for name in pdf_variants:
        copy_if_exists(HTML_ROOT / name, dest_root / name)

    # Mirror auxiliary assets (figures, notebooks, determinism stamps)
    assets_src = ASSETS_ROOT / source_slug
    if assets_src.exists():
        copy_if_exists(assets_src, dest_root / "assets")

    print(f"Publication assets for '{slug}' staged under {dest_root}")


def main() -> None:
    parser = argparse.ArgumentParser(description="Promote Omega portal artefacts to the public site")
    parser.add_argument("--slug", required=True, help="Publication slug (used on the public site)")
    parser.add_argument("--source-slug", help="Name of the build folder inside the Omega portal")
    parser.add_argument("--dest", default="public/publications", help="Destination root inside the repository")
    args = parser.parse_args()

    source_slug = args.source_slug or args.slug
    stage_publication(args.slug, source_slug, Path(args.dest))


if __name__ == "__main__":
    main()
