#!/usr/bin/env node
const { chromium } = require('playwright');
const path = require('path');
const fs = require('fs');
const { pathToFileURL } = require('url');
const { execSync } = require('child_process');

const [, , rawInput, outputPath, ...rest] = process.argv;

if (!rawInput || !outputPath) {
  console.error('Usage: node scripts/print.js <url> <output.pdf> [--format=Letter|A4] [--profile=release|dev]');
  process.exit(1);
}

let format = 'Letter';
let profile = 'release';
rest.forEach((arg) => {
  if (arg.startsWith('--format=')) {
    format = arg.split('=')[1] || format;
  }
  if (arg.startsWith('--profile=')) {
    profile = (arg.split('=')[1] || profile).toLowerCase();
  }
});

const targetUrl = /^https?:/i.test(rawInput) ? rawInput : pathToFileURL(path.resolve(rawInput)).href;

const buildInfo = {
  buildDate: new Date().toISOString(),
  commit: '',
  pageSize: format,
  profile
};

try {
  buildInfo.commit = execSync('git rev-parse --short HEAD', { stdio: ['ignore', 'pipe', 'ignore'] }).toString().trim();
} catch (error) {
  buildInfo.commit = 'unknown';
}

(async () => {
  const browser = await chromium.launch({ headless: true });
  const context = await browser.newContext();
  const page = await context.newPage();

  page.setDefaultTimeout(180000);
  page.setDefaultNavigationTimeout(180000);

  await page.addInitScript(({ profile, buildInfo }) => {
    window.__RFT_PRINT_PROFILE = profile;
    window.__RFT_BUILD_CONTEXT = buildInfo;
  }, { profile, buildInfo });

  await page.goto(targetUrl, { waitUntil: 'networkidle' });
  await page.emulateMedia({ media: 'print' });

  let ready = false;
  try {
    await page.waitForFunction(() => window.__RFT_PDF_READY === true, { timeout: 120000 });
    ready = true;
  } catch (error) {
    ready = await page.evaluate(() => window.__RFT_PDF_READY === true);
    if (!ready) {
      console.warn('Timed out waiting for window.__RFT_PDF_READY; proceeding with PDF export.');
    }
  }

  await fs.promises.mkdir(path.dirname(outputPath), { recursive: true });
  await page.pdf({
    path: outputPath,
    format,
    printBackground: true
  });

  await browser.close();
  console.log(`Saved PDF to ${outputPath}`);
})().catch((error) => {
  console.error(error);
  process.exit(1);
});
