#!/usr/bin/env python3
"""Package RC11 assets into a reproducible release bundle."""

from __future__ import annotations

import argparse
import hashlib
import json
import os
import subprocess
import tarfile
from datetime import datetime, UTC
from pathlib import Path
from typing import Iterable, List, Tuple
import sys


ROOT = Path(__file__).resolve().parents[1]
VALIDATION_SCRIPT = ROOT / "scripts" / "run_rc11_validation.py"
RC11_DIR = ROOT / "omega_document" / "vol1" / "dist" / "arxiv_rc11"
RC11_HTML = ROOT / "omega_document" / "admin_portal" / "static" / "html" / "volume1_rc11_body.html"
RC11_FIG_DIR = ROOT / "omega_document" / "admin_portal" / "static" / "vol1_rc11_figures"
RELEASE_DIR = ROOT / "releases"
TARBALL = RELEASE_DIR / "rft_volume1_rc11_release.tar.gz"
CHECKSUMS = RELEASE_DIR / "rft_volume1_rc11_sha256sums.txt"
RELEASE_MANIFEST = RELEASE_DIR / "rft_volume1_rc11_release.json"


def run_validation(skip_latex: bool = False) -> None:
    if not VALIDATION_SCRIPT.exists():
        raise FileNotFoundError(f"Validation script missing: {VALIDATION_SCRIPT}")
    python_bin = os.environ.get("RC11_VALIDATION_PYTHON", sys.executable)
    cmd = [python_bin, str(VALIDATION_SCRIPT)]
    if skip_latex:
        cmd.append("--skip-latex")
    subprocess.run(cmd, check=False, cwd=str(ROOT))


def iter_release_files() -> Iterable[Tuple[Path, Path]]:
    """Yield (absolute_path, archive_name)."""

    yield RC11_DIR / "main.tex", Path("rc11/main.tex")
    for tex in (RC11_DIR / "sections").rglob("*.tex"):
        yield tex, Path("rc11/sections") / tex.relative_to(RC11_DIR / "sections")
    yield RC11_DIR / "figures" / "manifest.csv", Path("rc11/figures/manifest.csv")
    if (RC11_DIR / "validation_report.json").exists():
        yield RC11_DIR / "validation_report.json", Path("rc11/reports/validation_report.json")
    if (RC11_DIR / "validation_report.md").exists():
        yield RC11_DIR / "validation_report.md", Path("rc11/reports/validation_report.md")
    if (RC11_DIR / "build.log").exists():
        yield RC11_DIR / "build.log", Path("rc11/logs/latex_build.log")
    if (RC11_DIR / "RFT_Vol1_v1.0_RC11_PREPRINT.pdf").exists():
        yield RC11_DIR / "RFT_Vol1_v1.0_RC11_PREPRINT.pdf", Path("rc11/RFT_Vol1_v1.0_RC11_PREPRINT.pdf")
    if RC11_HTML.exists():
        yield RC11_HTML, Path("web/volume1_rc11_body.html")
    if RC11_FIG_DIR.exists():
        for img in RC11_FIG_DIR.rglob("*"):
            if img.is_file():
                yield img, Path("web/vol1_rc11_figures") / img.relative_to(RC11_FIG_DIR)


def build_tarball(files: Iterable[Tuple[Path, Path]]) -> None:
    RELEASE_DIR.mkdir(parents=True, exist_ok=True)
    with tarfile.open(TARBALL, "w:gz") as archive:
        for abs_path, arc_path in files:
            archive.add(abs_path, arcname=str(arc_path))


def sha256(path: Path) -> str:
    digest = hashlib.sha256()
    with path.open("rb") as handle:
        for chunk in iter(lambda: handle.read(1024 * 1024), b""):
            digest.update(chunk)
    return digest.hexdigest()


def write_checksums(extra_files: List[Path]) -> None:
    lines: List[str] = []
    for path in [TARBALL] + extra_files:
        if path.exists():
            lines.append(f"{sha256(path)}  {path.relative_to(ROOT)}")
    CHECKSUMS.write_text("\n".join(lines) + "\n", encoding="utf-8")


def write_manifest() -> None:
    metadata = {
        "generated_at": datetime.now(UTC).isoformat(),
        "tarball": str(TARBALL.relative_to(ROOT)),
        "checksums": str(CHECKSUMS.relative_to(ROOT)),
        "latex_source": str((RC11_DIR / "main.tex").relative_to(ROOT)),
        "html_body": str(RC11_HTML.relative_to(ROOT)) if RC11_HTML.exists() else None,
        "figures_dir": str(RC11_FIG_DIR.relative_to(ROOT)) if RC11_FIG_DIR.exists() else None,
        "git_commit": subprocess.run(
            ["git", "rev-parse", "HEAD"],
            cwd=str(ROOT),
            check=False,
            stdout=subprocess.PIPE,
            text=True,
        ).stdout.strip(),
    }
    RELEASE_MANIFEST.write_text(json.dumps(metadata, indent=2) + "\n", encoding="utf-8")


def main() -> int:
    parser = argparse.ArgumentParser(description=__doc__)
    parser.add_argument("--skip-validation", action="store_true", help="Do not re-run validation script")
    parser.add_argument("--skip-latex", action="store_true", help="Skip LaTeX compile when running validation")
    args = parser.parse_args()

    if not args.skip_validation:
        run_validation(skip_latex=args.skip_latex)

    files = list(iter_release_files())
    build_tarball(files)

    extras = [RC11_HTML, RC11_DIR / "RFT_Vol1_v1.0_RC11_PREPRINT.pdf"]
    write_checksums(extras)
    write_manifest()

    print(f"Created tarball: {TARBALL}")
    print(f"Checksums written to: {CHECKSUMS}")
    print(f"Release manifest: {RELEASE_MANIFEST}")
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
