#!/usr/bin/env node
const { chromium } = require('playwright');
const path = require('path');
const { pathToFileURL } = require('url');

const args = process.argv.slice(2);
const rawInput = args[0];
const options = args.slice(1);

if (!rawInput) {
  console.error('Usage: node scripts/preflight.js <url> [--profile=release|dev]');
  process.exit(1);
}

let profile = 'release';
options.forEach((arg) => {
  if (arg.startsWith('--profile=')) {
    profile = (arg.split('=')[1] || profile).toLowerCase();
  }
});

const targetUrl = /^https?:/i.test(rawInput) ? rawInput : pathToFileURL(path.resolve(rawInput)).href;

(async () => {
  const browser = await chromium.launch({ headless: true });
  const context = await browser.newContext();
  const page = await context.newPage();

  page.setDefaultTimeout(180000);
  page.setDefaultNavigationTimeout(180000);

  await page.addInitScript(({ profile }) => {
    window.__RFT_PRINT_PROFILE = profile;
  }, { profile });

  await page.goto(targetUrl, { waitUntil: 'networkidle' });
  await page.emulateMedia({ media: 'print' });

  let ready = false;
  try {
    await page.waitForFunction(() => window.__RFT_PDF_READY === true, { timeout: 120000 });
    ready = true;
  } catch (error) {
    ready = await page.evaluate(() => window.__RFT_PDF_READY === true);
    if (!ready) {
      console.warn('Timed out waiting for window.__RFT_PDF_READY; continuing with DOM inspection.');
    }
  }

  const results = await page.evaluate(() => {
    const h1Count = Array.from(document.querySelectorAll('#content > section > h1')).length;
    const h2Count = Array.from(document.querySelectorAll('#content > section h2')).length;
    const tocH1 = Array.from(document.querySelectorAll('#toc .toc-item--h1 > a')).length;
    const tocH2 = Array.from(document.querySelectorAll('#toc .toc-item--h2 > a')).length;
    const figureCount = Array.from(document.querySelectorAll('figure')).length;
    const tableCount = Array.from(document.querySelectorAll('table')).length;
    const lofCount = Array.from(document.querySelectorAll('#lof .catalog-item')).length;
    const lotCount = Array.from(document.querySelectorAll('#lot .catalog-item')).length;
    const missingMarkers = Array.from(document.querySelectorAll('body'))
      .some((node) => /\[missing\s.+\]/i.test(node.textContent));
    const unresolvedXrefs = Array.from(document.querySelectorAll('a.xref'))
      .filter((link) => {
        const href = link.getAttribute('href');
        return href && href.startsWith('#') && !document.querySelector(href);
      })
      .map((link) => link.getAttribute('href'));
    const imagesMissingAlt = Array.from(document.querySelectorAll('img'))
      .filter((img) => !(img.getAttribute('alt') || '').trim())
      .map((img) => img.getAttribute('src') || img.currentSrc || '[inline image]');

    return {
      h1Count,
      h2Count,
      tocH1,
      tocH2,
      figureCount,
      tableCount,
      lofCount,
      lotCount,
      missingMarkers,
      unresolvedXrefs,
      imagesMissingAlt,
      readyFlag: window.__RFT_PDF_READY === true
    };
  });

  await browser.close();

  const failures = [];

  if (results.h1Count !== results.tocH1) {
    failures.push(`H1 count (${results.h1Count}) != ToC H1 entries (${results.tocH1})`);
  }
  if (results.h2Count !== results.tocH2) {
    failures.push(`H2 count (${results.h2Count}) != ToC H2 entries (${results.tocH2})`);
  }
  if (results.figureCount !== results.lofCount) {
    failures.push(`Figure count (${results.figureCount}) != LoF entries (${results.lofCount})`);
  }
  if (results.tableCount !== results.lotCount) {
    failures.push(`Table count (${results.tableCount}) != LoT entries (${results.lotCount})`);
  }
  if (results.missingMarkers) {
    failures.push('Detected "[missing …]" placeholders in document.');
  }
  if (results.unresolvedXrefs.length) {
    failures.push(`Unresolved cross references: ${results.unresolvedXrefs.join(', ')}`);
  }
  if (results.imagesMissingAlt.length) {
    failures.push(`Images missing alt text: ${results.imagesMissingAlt.join(', ')}`);
  }
  if (!ready && !results.readyFlag) {
    failures.push('Document readiness flag remained false.');
  }

  if (failures.length) {
    console.error('Preflight failed:\n - ' + failures.join('\n - '));
    process.exit(1);
  }

  console.log('Preflight checks passed.');
})();
