#!/usr/bin/env python3
"""
C10.2 Decision Gate - Instant GO/MAYBE/NO-GO verdict after grid search.

Usage:
    python3 scripts/post_c10p_gate.py --summary reports/_summary_C10P_TRAIN.json
"""
import json
import sys
import math
import argparse


def wilson_ci(k, n, alpha=0.05):
    """Wilson score confidence interval."""
    if n == 0:
        return 0.0, 0.0
    z = 1.96  # 95% CI
    p_hat = k / n
    d = z * z
    denom = 1 + d / n
    center = (p_hat + d / (2 * n)) / denom
    margin = (
        z * math.sqrt(p_hat * (1 - p_hat) / n + d / (4 * n * n)) / denom
    )
    return max(0, center - margin), min(1, center + margin)


def main():
    parser = argparse.ArgumentParser(
        description="C10.2 decision gate - GO/MAYBE/NO-GO verdict"
    )
    parser.add_argument(
        "--summary",
        default="reports/_summary_C10P_TRAIN.json",
        help="Summary JSON from best C10.2 config",
    )
    parser.add_argument(
        "--pass-threshold",
        type=float,
        default=20.0,
        help="RMS% threshold for pass/fail",
    )
    parser.add_argument(
        "--test-n", type=int, default=34, help="Expected TEST cohort size"
    )
    args = parser.parse_args()

    try:
        data = json.load(open(args.summary))
    except FileNotFoundError:
        print(f"❌ Summary file not found: {args.summary}")
        sys.exit(1)

    rows = data.get("rows", [])
    n = len(rows)

    if n == 0:
        print(f"❌ No galaxies in summary: {args.summary}")
        sys.exit(1)

    # Count total passes
    k = sum(
        1
        for r in rows
        if r.get("solvers", {})
        .get("rft_geom", {})
        .get("rms_pct", float("inf"))
        <= args.pass_threshold
    )

    # Wilson 95% CI
    p_hat = k / n
    lo, hi = wilson_ci(k, n)
    loN, hiN = int(args.test_n * lo), int(args.test_n * hi)

    # Subgroup analysis
    def tag_pass(tag):
        return sum(
            1
            for r in rows
            if (tag in r.get("tags", []))
            and (
                r.get("solvers", {})
                .get("rft_geom", {})
                .get("rms_pct", float("inf"))
                <= args.pass_threshold
            )
        )

    def tag_total(tag):
        return sum(1 for r in rows if tag in r.get("tags", []))

    lsb_k, lsb_n = tag_pass("LSB"), tag_total("LSB")
    hsb_k, hsb_n = tag_pass("HSB"), tag_total("HSB")

    # Median RMS
    med = data.get("solver_totals", {}).get("rft_geom", {}).get("rms_pct_median", 0.0)

    # Print results
    print("=" * 70)
    print("C10.2 POWER-TAIL GATE DECISION")
    print("=" * 70)
    print(
        f"TRAIN: {k}/{n} @ {args.pass_threshold:.0f}% → {100*p_hat:.1f}% pass rate"
    )
    print(f"Wilson 95% CI: [{100*lo:.1f}%, {100*hi:.1f}%]")
    print(f"Median RMS: {med:.2f}%")
    print(f"\nSubgroup breakdown:")
    print(f"  LSB: {lsb_k}/{lsb_n} ({100*lsb_k/max(lsb_n, 1):.1f}%)")
    print(f"  HSB: {hsb_k}/{hsb_n} ({100*hsb_k/max(hsb_n, 1):.1f}%)")
    print(f"\nPredicted TEST range (n={args.test_n}): [{loN}, {hiN}] passes")
    print("=" * 70)

    # Decision rubric
    decision = "NO-GO"
    color = "🔴 RED ZONE"

    # GREEN: ≥40% pass rate OR ≥30% with strong LSB lift
    if p_hat >= 0.40:
        decision = "GO"
        color = "🟢 GREEN ZONE"
        action = "Freeze config, pre-register, run full TEST validation"
    elif p_hat >= 0.30:
        # Check for LSB lift (at least 40% of LSB galaxies pass)
        lsb_rate = lsb_k / max(lsb_n, 1)
        if lsb_rate >= 0.4:
            decision = "GO (marginal)"
            color = "🟢 GREEN ZONE (marginal)"
            action = "Strong LSB lift observed. Proceed to TEST with caution."
        else:
            decision = "MAYBE"
            color = "🟡 YELLOW ZONE"
            action = "Consider micro-tune: lower A_flat or widen sigma_ln_r"
    # YELLOW: 20-30% pass rate
    elif p_hat >= 0.20:
        decision = "MAYBE"
        color = "🟡 YELLOW ZONE"
        action = "Try one micro-tune iteration, then re-gate. If still <30%, stop."
    # RED: <20% pass rate
    else:
        decision = "NO-GO"
        color = "🔴 RED ZONE"
        action = "Optional: Run C11 kernel-mix test (9 configs). Otherwise, publish negative result."

    print(f"\n{color}")
    print(f"DECISION: {decision}")
    print(f"ACTION: {action}")
    print("=" * 70)

    # Detailed recommendations
    if decision == "GO" or decision == "GO (marginal)":
        print("\n📋 Next steps for GO:")
        print("  1. cp config/global_c10p_best.json config/global_c10p_tail.json")
        print("  2. sha256sum config/global_c10p_tail.json > config/global_c10p_tail.json.sha256")
        print("  3. Create scripts/PRE_REGISTRATION_C10.md with Wilson CI")
        print("  4. Run TEST validation for rft_geom, mond, nfw_fit")
        print("  5. Aggregate and export figures")

    elif decision == "MAYBE":
        print("\n📋 Micro-tune options (pick ONE):")
        print("  Option A: Soften inner (if r_turn=0 in best config)")
        print("    - Keep best A0, alpha")
        print("    - Set r_turn_kpc=2.0, A_flat=0.28")
        print("  Option B: De-ripple (if sigma_ln_r=0.50)")
        print("    - Keep best A0, alpha, A_flat")
        print("    - Set sigma_ln_r=0.55")
        print("\n  Then re-run gate. If still <30%, stop.")

    else:  # NO-GO
        print("\n📋 Final validation options:")
        print("  Option 1: Run C11 kernel-mix test (9 configs, ~30 mins)")
        print("    - Test Gaussian-Lorentzian kernel blend")
        print("    - If still <25%, close as negative result")
        print("  Option 2: Skip C11, publish comprehensive negative result now")
        print("    - C8/C8.2: 14.5% (parameter tuning exhausted)")
        print("    - C9: 12.7% (descriptor mapping falsified)")
        print("    - C10: 14.5% (constant-v tail)")
        print("    - C10.2: XX% (power-law tail)")
        print("    - Conclusion: RFT rotation curves have fundamental inadequacy")

    print("=" * 70)

    # Exit code for scripting
    if decision in ("GO", "GO (marginal)"):
        sys.exit(0)  # Success - proceed to TEST
    elif decision == "MAYBE":
        sys.exit(1)  # Try micro-tune
    else:
        sys.exit(2)  # Stop here or run C11


if __name__ == "__main__":
    main()
