#!/bin/bash
# RFT Cosmology Site - Nightly Backup System
# Creates incremental git backups with rotation

set -e

# Configuration
BACKUP_DIR="/home/rftuser/backups"
REPO_DIR="/home/rftuser/rft-cosmology-site"
DATE=$(date +%Y%m%d_%H%M%S)
KEEP_DAYS=30  # Keep backups for 30 days

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

log() {
    echo -e "${GREEN}[$(date '+%Y-%m-%d %H:%M:%S')]${NC} $1"
}

error() {
    echo -e "${RED}[$(date '+%Y-%m-%d %H:%M:%S')] ERROR:${NC} $1" >&2
}

warning() {
    echo -e "${YELLOW}[$(date '+%Y-%m-%d %H:%M:%S')] WARNING:${NC} $1"
}

# Create backup directory
mkdir -p "$BACKUP_DIR"

cd "$REPO_DIR"

# Check if we're in a git repo
if [ ! -d ".git" ]; then
    error "Not in a git repository"
    exit 1
fi

log "Starting nightly backup for RFT Cosmology Site"

# Create backup filename
BACKUP_NAME="rft-site-backup-${DATE}"
BACKUP_PATH="$BACKUP_DIR/$BACKUP_NAME"

# 1. Git Bundle Backup (most important - captures ALL history)
log "Creating git bundle backup..."
git bundle create "$BACKUP_PATH.bundle" --all
if [ $? -eq 0 ]; then
    log "Git bundle created: $BACKUP_PATH.bundle"
else
    error "Failed to create git bundle"
    exit 1
fi

# 2. Full Archive Backup (includes untracked files)
log "Creating full archive backup..."
tar -czf "$BACKUP_PATH.tar.gz" \
    --exclude='.git' \
    --exclude='node_modules' \
    --exclude='__pycache__' \
    --exclude='*.pyc' \
    --exclude='.venv' \
    --exclude='venv' \
    --exclude='out/' \
    --exclude='work/' \
    .

if [ $? -eq 0 ]; then
    log "Archive created: $BACKUP_PATH.tar.gz"
else
    error "Failed to create archive"
    exit 1
fi

# 3. Database Backup (if exists)
if [ -f "instance/rft.sqlite" ]; then
    log "Backing up database..."
    cp "instance/rft.sqlite" "$BACKUP_PATH.sqlite"
fi

# 4. Create manifest file
log "Creating backup manifest..."
cat > "$BACKUP_PATH.manifest" << EOF
# RFT Cosmology Site Backup Manifest
# Created: $(date)
# Host: $(hostname)
# Git Commit: $(git rev-parse HEAD 2>/dev/null || echo "unknown")
# Git Branch: $(git branch --show-current 2>/dev/null || echo "unknown")

Files in this backup:
- ${BACKUP_NAME}.bundle (Git repository with full history)
- ${BACKUP_NAME}.tar.gz (Full site archive excluding .git)
- ${BACKUP_NAME}.sqlite (Database backup, if exists)
- ${BACKUP_NAME}.manifest (This file)

To restore:
1. Git restore: git clone ${BACKUP_NAME}.bundle restored-repo
2. Files restore: tar -xzf ${BACKUP_NAME}.tar.gz
3. Database restore: cp ${BACKUP_NAME}.sqlite instance/rft.sqlite

Backup size:
$(du -sh "$BACKUP_PATH"* | head -10)
EOF

# 5. Cleanup old backups
log "Cleaning up old backups (keeping last $KEEP_DAYS days)..."
find "$BACKUP_DIR" -name "rft-site-backup-*" -type f -mtime +$KEEP_DAYS -delete
removed=$(find "$BACKUP_DIR" -name "rft-site-backup-*" -type f -mtime +$KEEP_DAYS 2>/dev/null | wc -l)
if [ $removed -gt 0 ]; then
    log "Removed $removed old backup files"
fi

# 6. Verify backup integrity
log "Verifying backup integrity..."
if git bundle verify "$BACKUP_PATH.bundle" > /dev/null 2>&1; then
    log "✅ Git bundle verification passed"
else
    error "❌ Git bundle verification failed"
    exit 1
fi

if tar -tzf "$BACKUP_PATH.tar.gz" > /dev/null 2>&1; then
    log "✅ Archive verification passed"
else
    error "❌ Archive verification failed"
    exit 1
fi

# 7. Summary
BUNDLE_SIZE=$(du -sh "$BACKUP_PATH.bundle" | cut -f1)
ARCHIVE_SIZE=$(du -sh "$BACKUP_PATH.tar.gz" | cut -f1)
TOTAL_BACKUPS=$(ls -1 "$BACKUP_DIR"/rft-site-backup-*.bundle 2>/dev/null | wc -l)

log "🎉 Backup completed successfully!"
log "📁 Bundle size: $BUNDLE_SIZE"
log "📦 Archive size: $ARCHIVE_SIZE" 
log "📊 Total backups stored: $TOTAL_BACKUPS"
log "💾 Backup location: $BACKUP_DIR"

# Optional: Send notification (uncomment if you want email alerts)
# echo "RFT Site backup completed successfully at $(date)" | mail -s "Backup Success" your-email@domain.com