#!/usr/bin/env python3
"""Generate ablation configs from frozen v2 config to test causal components."""
import json
import os
import copy
import sys
from pathlib import Path

base = Path("config/global_rc_v2_frozen.json")
f = json.load(open(base))
outdir = Path("config/ablations")
outdir.mkdir(parents=True, exist_ok=True)

def dump(tag, cfg):
    p = outdir / f"rc_v2_{tag}.json"
    json.dump(cfg, open(p, "w"), indent=2)
    print(f"Created: {p}")
    return p

# Extract tail config from frozen
tail = f["tail"]

# 1) Tail OFF (Newtonian baseline)
c = copy.deepcopy(f)
c["tail"]["A0_kms2_per_kpc"] = 0.0
c["tail"]["comment"] = "Ablation: tail OFF (Newtonian baseline)"
dump("tail_off", c)

# 2) No acceleration gate (always-on tail)
c = copy.deepcopy(f)
c["tail"]["gamma"] = 0.0
c["tail"]["g_star_kms2_per_kpc"] = 1e99
c["tail"]["comment"] = "Ablation: no acceleration gate (tail always on)"
dump("no_g_gate", c)

# 3) No radial onset (tail active everywhere)
c = copy.deepcopy(f)
c["tail"]["r_turn_kpc"] = 0.0
c["tail"]["comment"] = "Ablation: no radial onset gate"
dump("no_r_gate", c)

# 4) alpha = 1.0 (constant-velocity tail)
c = copy.deepcopy(f)
c["tail"]["alpha"] = 1.0
c["tail"]["comment"] = "Ablation: alpha=1.0 (constant-velocity tail)"
dump("alpha1_const_v", c)

# 5) Soften gate (gamma * 0.75)
c = copy.deepcopy(f)
c["tail"]["gamma"] = 0.75 * float(tail["gamma"])
c["tail"]["comment"] = f"Ablation: softer gate (gamma={c['tail']['gamma']:.3f})"
dump("gate_soft", c)

print(f"\nCreated 5 ablation configs in {outdir}/")
print("Next: Run each config on TRAIN/TEST and compare to frozen baseline")
