#!/usr/bin/env python3
"""
Script to convert all RFT papers into YAML files.
This version uses a more robust block splitter that works with both:
- Standard YAML separators (---)
- Comment header lines (# ------ RFT x.y ------)
- Blocks that start with 'slug:' directly
"""

import os
import pathlib
import re
import traceback

# Enable this for extra debug information
DEBUG = True

# Create directories if they don't exist
def ensure_dirs():
    root = pathlib.Path("app/papers")
    featured_dir = root / "featured"
    archive_dir = root / "archive"
    
    for directory in [root, featured_dir, archive_dir]:
        directory.mkdir(parents=True, exist_ok=True)
    
    return root, featured_dir, archive_dir

# Clean YAML block by removing contentReference tags and other problematic elements
def clean_yaml_block(block):
    # Remove contentReference tags
    block = re.sub(r':contentReference\[oaicite:\d+\]\{index=\d+\}', '', block)
    
    # Fix LaTeX escape sequences in citations
    block = block.replace(r'\&', r'&')
    block = block.replace(r'\emph', r'emph')  # Fix invalid escape sequence warnings
    
    # Make sure multiline strings use proper YAML syntax
    lines = []
    in_multiline = False
    
    for line in block.split('\n'):
        # Skip separator lines and empty lines at the beginning
        if line.strip() == '---' or (not lines and not line.strip()):
            continue
            
        # Skip comment lines at the beginning of a block
        if not lines and line.strip().startswith('#'):
            continue
            
        # Handle abstract fields
        if 'abstract:' in line and '>' not in line:
            line = line.replace('abstract:', 'abstract: >')
            in_multiline = True
        
        # Handle citation fields
        elif 'citation:' in line and '|' not in line:
            line = line.replace('citation:', 'citation: |')
            in_multiline = True
        
        # Add proper indentation to multiline content
        elif in_multiline and line.strip() and not line.strip().startswith('-') and ':' not in line:
            if not line.startswith('  '):
                line = '  ' + line
        elif line.strip() and ':' in line:
            in_multiline = False
        
        lines.append(line)
    
    return '\n'.join(lines)

# More accurately split YAML blocks - this is the key improved function
def split_yaml_blocks(yaml_data):
    # We'll split on:
    # 1. Actual YAML document separators (---)
    # 2. Comment lines with RFT version headers (# ----- RFT x.y ----)
    # 3. Lines that start with "slug:" after a newline
    
    # First, normalize newlines
    yaml_data = yaml_data.replace('\r\n', '\n')
    
    # Pattern looks for either:
    # - A line with "---" and optional whitespace
    # - A line starting with "# -" followed by dashes and RFT
    # - A line starting with "slug:" (with a newline before it)
    pattern = r'\n(?:---|\# -+\s*RFT[^-]*-+|slug:)'
    
    # Add a newline at the beginning to handle the first block
    yaml_data = '\n' + yaml_data
    
    # Find all split positions
    matches = list(re.finditer(pattern, yaml_data))
    
    if DEBUG:
        print(f"[DEBUG] Found {len(matches)} potential block separators")
    
    # If no separators found, return the whole thing as one block
    if not matches:
        return [yaml_data.strip()]
    
    # Extract blocks using the split positions
    blocks = []
    for i in range(len(matches)):
        start_pos = matches[i].start()
        
        # If it's a slug line (not a separator or comment line), we need to include "slug:" in the block
        is_slug_line = matches[i].group().strip() == 'slug:'
        
        # Adjust start position to include "slug:" if needed
        if is_slug_line:
            start_pos += 1  # Skip just the newline
        
        # For the end position, use the start of the next match or the end of the string
        end_pos = matches[i+1].start() if i < len(matches) - 1 else len(yaml_data)
        
        # Extract the block
        block = yaml_data[start_pos:end_pos].strip()
        
        # Skip empty blocks
        if block:
            blocks.append(block)
    
    if DEBUG:
        print(f"[DEBUG] Split into {len(blocks)} non-empty blocks")
        for i, block in enumerate(blocks[:2]):
            print(f"[DEBUG] Block #{i+1} preview: {block[:50].replace(chr(10), ' ')}...")
        if len(blocks) > 2:
            print(f"[DEBUG] ... and {len(blocks) - 2} more blocks")
    
    return blocks

# Find the slug in a YAML block
def find_slug(block):
    # Try direct pattern matching first
    slug_match = re.search(r'^slug:\s*(.+)$', block, re.MULTILINE)
    if slug_match:
        return slug_match.group(1).strip()
    
    # Try looking for a line that starts with slug: (with potential whitespace)
    lines = block.splitlines()
    for line in lines:
        line = line.strip()
        if line.startswith('slug:'):
            parts = line.split(':', 1)
            if len(parts) > 1:
                return parts[1].strip()
    
    return None

# Process and write the YAML blocks to files
def process_yaml():
    root, featured_dir, archive_dir = ensure_dirs()
    
    # Check for existing files to avoid duplicates
    existing_slugs = set()
    for folder in [featured_dir, archive_dir]:
        for file in folder.glob("*.yml"):
            existing_slugs.add(file.stem)
    
    if DEBUG:
        print(f"[DEBUG] Found {len(existing_slugs)} existing YML files")
        if existing_slugs:
            print(f"[DEBUG] Examples: {list(existing_slugs)[:3]}...")
    
    # Split the content into individual YAML blocks
    docs = split_yaml_blocks(YAML_SNIPPETS)
    
    if DEBUG:
        print(f"[DEBUG] Found {len(docs)} blocks in the YAML data")
    
    success_count = 0
    error_count = 0
    skipped_count = 0
    
    for i, doc in enumerate(docs):
        if not doc:
            if DEBUG:
                print(f"[DEBUG] Block #{i+1} is empty, skipping")
            skipped_count += 1
            continue
        
        if DEBUG:
            print(f"[DEBUG] Processing block #{i+1}, length: {len(doc)} chars")
            print(f"[DEBUG] First 50 chars: {doc[:50].replace(chr(10), ' ')}")
        
        try:
            # Clean up the block
            clean_doc = clean_yaml_block(doc)
            
            # Try to find the slug
            slug = find_slug(clean_doc)
            
            if not slug:
                print(f"⚠️ Block #{i+1} has no slug field, skipping")
                if DEBUG:
                    print(f"[DEBUG] Block content preview:\n{clean_doc[:200]}")
                error_count += 1
                continue
            
            # Skip if already processed in previous run
            if slug in existing_slugs:
                print(f"⚠️ Slug '{slug}' already exists, skipping")
                skipped_count += 1
                continue
            
            # Handle duplicate slugs within this batch
            duplicate_count = 1
            original_slug = slug
            while slug in existing_slugs:
                duplicate_count += 1
                slug = f"{original_slug}-v{duplicate_count}"
                
            if slug != original_slug:
                print(f"⚠️ Changed duplicate slug '{original_slug}' to '{slug}'")
                clean_doc = re.sub(r'(^slug:\s*)' + re.escape(original_slug), r'\1' + slug, clean_doc, flags=re.MULTILINE)
            
            # Check for featured status
            featured = 'featured: true' in clean_doc.lower()
            folder = featured_dir if featured else archive_dir
            
            # Write the file
            output_file = folder / f"{slug}.yml"
            output_file.write_text(clean_doc + "\n", encoding="utf-8")
            
            print(f"✓ Wrote {slug}.yml -> {folder}")
            success_count += 1
            existing_slugs.add(slug)  # Add to our tracking set
            
        except Exception as e:
            print(f"❌ Error processing block #{i+1}: {str(e)}")
            if DEBUG:
                traceback.print_exc()
            error_count += 1
    
    print(f"\nProcessing complete: {success_count} files created, {error_count} errors, {skipped_count} blocks skipped")

# The YAML snippets - Include ALL papers
YAML_SNIPPETS = """
# --- RFT 10.0 ---
slug: rft-10-0-rc1
title: "Final Relativistic Field Theory Framework"
version: "10.0 RC1"
date: 2025-02-10
tags: ["Foundations", "Entropy-Time", "Scalaron EoM"]
abstract: >
  Locks in ...
key_results:
  - Definitive scalaron equation couples ...
read_link: "/static/docs/pdfs/rft-10-0-rc1.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2010.0%20RC1.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT10,
    ...
  }
featured: true

# --- RFT 10.1 ---
slug: rft-10-1
title: "Gauge Theory Extensions"
featured: true

# keep pasting every block, separate with `---`
slug: rft-10-0-rc1-v2
title: "Relativistic Field Theory 10.0 – Emergent-Time & Scalaron Framework"
version: "10.0 RC1"
date: 2025-02-15          # adjust if you prefer the exact commit date
tags: ["Foundations", "Emergent-Time", "Scalaron"]
abstract: >
  Finalizes the core RFT equations by locking in the scalaron's curvature-
  and matter-coupled dynamics, then demonstrates how time itself emerges
  as an entropy functional of the field.  RFT 10.0 shows that the same
  scalaron framework reproduces ΛCDM limits, fuzzy-DM cores, and modified
  gravity in different regimes, yielding crisp, falsifiable thresholds
  for core collapse, lensing flicker, and gravitational-wave entropy.
key_results:
  - Locks the complete scalaron field equation with non-redundant curvature (α) and matter (β) couplings
  - Defines time as ΔS of the scalaron, embedding the thermodynamic arrow directly in field evolution
  - Maps RFT onto ΛCDM, fuzzy-DM, and modified-gravity limits and lists testable core-collapse & lensing thresholds
read_link: "/static/docs/pdfs/rft-10-0-rc1.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2010.0%20RC1.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT10,
    author  = {Fitzpatrick, I.},
    title   = {Relativistic Field Theory 10.0 – Emergent-Time & Scalaron Framework},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 10.0 RC1}
  }
featured: true

# ───────────────────────────  RFT 10.1  ───────────────────────────
slug: rft-10-1-full
title: "Gauge-Bridge: Emergent U(1) Electrodynamics"
version: "10.1 RC0"
date: 2025-02-15
tags: ["Electromagnetism", "Gauge-Bridge"]
abstract: >
  Extends RFT by promoting the scalaron's global phase to a local U(1)
  symmetry, introducing an emergent, massless spin-1 gauge field that
  reproduces Maxwell's equations in the weak-field limit.
key_results:
  - Derives fully gauge-invariant action and field equations for ϕ and Aμ
  - Identifies scalaron vortices as quantized charge/flux carriers
  - One-loop β-function matches QED, giving α ≃ 1 / 137 at low energy
read_link: "/static/docs/pdfs/RFT%2010.1.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2010.1.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT10_1,
    author  = {Fitzpatrick, I.},
    title   = {Gauge-Bridge: Emergent U(1) Electrodynamics},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 10.1 RC0}
  }
featured: true

# ───────────────────────────  RFT 10.2  ───────────────────────────
slug: rft-10-2
title: "Scalaron Triplet & Emergent SU(2) Weak Sector"
version: "10.2"
date: 2025-02-24
tags: ["Weak-Force", "SU(2)"]
abstract: >
  Shows that promoting the scalaron to an adjoint triplet with local
  SU(2) symmetry yields a Georgi–Glashow–type gauge field that is
  asymptotically free and supports 't Hooft–Polyakov monopoles.
key_results:
  - Demonstrates SU(2) gauge field emergence via twistor triplet bundle
  - Predicts finite-energy monopoles at M ≈ 𝒪(TeV) consistent with null searches
  - One-loop β-function (b₀ = 7) confirms asymptotic freedom
read_link: "/static/docs/pdfs/RFT%2010.2.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2010.2.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT10_2,
    author  = {Fitzpatrick, I.},
    title   = {Scalaron Triplet and Emergent SU(2) Weak Sector},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 10.2}
  }
featured: true

# ---------- RFT 10.4 ----------
slug: rft-10-4
title: "Fermionic Spectrum from Twistor–Scalaron Topological Emergence"
version: "10.4"
date: 2025-02-10            # adjust if repo metadata differs
tags: ["Fermions", "Topology", "Foundations"]
abstract: >
  Shows that all Standard-Model fermions, their threefold family
  replication, and precise gauge charges arise as topological
  excitations of a single scalaron–twistor bundle, replacing ad-hoc
  matter content with geometry-driven modes.
key_results:
  - Standard-Model fermions emerge as twistor–scalaron zero-modes, not inserted by hand
  - Winding-number 3 of the bundle traps exactly three chiral families
  - Anomaly cancellation and charge quantization follow automatically from bundle topology
read_link: "/static/docs/pdfs/rft-10-4.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2010.4.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT10_4,
    author  = {Fitzpatrick, I.},
    title   = {Fermionic Spectrum from Twistor–Scalaron Topological Emergence},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 10.4}
  }
featured: true

# ---------- RFT 10.5 ----------
slug: rft-10-5
title: "Emergence of Electroweak Symmetry from Twistor–Scalaron Geometry"
version: "10.5"
date: 2025-02-22
tags: ["Electroweak", "Hypercharge"]
abstract: >
  Derives the full SU(2)_L × U(1)_Y gauge sector as a holomorphic
  symmetry of the scalaron–twistor bundle, then reproduces electroweak
  breaking, particle masses and the Weinberg angle with percent-level
  accuracy while explaining hypercharge quantization.
key_results:
  - SU(2)_L and U(1)_Y appear as bundle holonomies, no external input
  - Correct M_W, M_Z and sin²θ_W emerge from scalaron VEV dynamics
  - Hypercharge pattern & anomaly cancellation traced to twistor topology
read_link: "/static/docs/pdfs/rft-10-5.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2010.5.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT10_5,
    author  = {Fitzpatrick, I.},
    title   = {Emergence of Electroweak Symmetry from Twistor–Scalaron Geometry},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 10.5}
  }
featured: true

# ---------- RFT 10.6 ----------
slug: rft-10-6
title: "Planck-Scale UV Completion in the Scalaron–Twistor Framework"
version: "10.6"
date: 2025-03-01
tags: ["Quantum-Gravity", "UV-Completion"]
abstract: >
  Extends RFT into a fully quantized, asymptotically safe gravity at
  M_Pl, showing how classical spacetime, black-hole unitarity and
  Big-Bang singularity resolution arise from fuzzy twistor geometry, and
  outlining observational signatures from CMB anomalies to GW echoes.
key_results:
  - Establishes UV-finite fixed point linking twistor theory and LQG
  - Replaces Big-Bang & black-hole singularities with quantum bounces keeping information unitary
  - Predicts CMB imprints and gravitational-wave echoes as near-term tests
read_link: "/static/docs/pdfs/rft-10-6.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2010.6.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT10_6,
    author  = {Fitzpatrick, I.},
    title   = {Planck-Scale UV Completion in the Scalaron–Twistor Framework},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 10.6}
  }
featured: true


# ---------- RFT 11.0 ----------
slug: rft-11-0
title: "Memory-Bound Scalaron: A Unified Field Framework"
version: "11.0"
date: 2025-03-28
tags: ["Foundations", "Phenomenology"]
abstract: >
  Presents the complete Relativistic Field Theory (RFT) in which a single
  scalaron field, coupled to twistor geometry, yields spacetime, gravity,
  Standard-Model forces and an entropy-defined arrow of time. The theory
  resolves the core–cusp and missing-satellite problems, reproduces MOND-like
  galaxy dynamics without extra fields, and remains consistent with Solar-System
  tests.
key_results:
  - Derives GR + SM limits from a single scalaron–twistor action
  - Predicts dwarf-galaxy core density and MOND scale \(a_0\) from first principles
  - Unifies dark matter & dark energy behavior in one adaptive field
read_link: "/static/docs/pdfs/rft-11-0.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2011.0.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT11,
    author  = {Fitzpatrick, I.},
    title   = {Memory-Bound Scalaron: A Unified Field Framework},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 11.0}
  }
featured: true

# ---------- RFT 11.1 ----------
slug: rft-11-1
title: "Scalaron–Twistor Precision Fit to Standard-Model Data"
version: "11.1"
date: 2025-04-07
tags: ["Phenomenology", "Particle Physics"]
abstract: >
  Benchmarks the scalaron–twistor framework against Particle-Data-Group
  observables. Using geometric overlap integrals only, it reproduces all
  quark- and lepton-masses, CKM and PMNS mixing angles, and electroweak
  couplings within one percent of experiment, while maintaining gauge-anomaly
  cancellation and custodial symmetry.
key_results:
  - Achieves ≤1 % agreement for 12 fermion masses & Yukawas, plus CKM/PMNS angles
  - Predicts \( \alpha_s,\; \alpha_{\mathrm{EM}},\; \sin^2\theta_W \) at \(M_Z\) inside 0.1 %
  - Confirms full gauge & gravitational anomaly cancellation in the unified action
read_link: "/static/docs/pdfs/rft-11-1.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2011.1.docx"
supplementary:
  - label: "Yukawa-fit notebook"
    file:  "rft11_1_yukawa.ipynb"
citation: |
  @article{Fitzpatrick2025RFT11_1,
    author  = {Fitzpatrick, I.},
    title   = {Scalaron–Twistor Precision Fit to Standard-Model Data},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 11.1}
  }
featured: true

# ---------- RFT 11.2 ----------
slug: rft-11-2
title: "FRG & BRST Consistency of the Scalaron–Twistor Theory"
version: "11.2"
date: 2025-04-18
tags: ["Quantum Consistency", "Renormalization"]
abstract: >
  Carries out a full quantum-field audit of RFT. Functional Renormalization-Group
  equations reveal an interacting UV fixed point, rendering the theory
  asymptotically safe. A BRST quantization shows the ghost spectrum cancels
  anomalies, while higher-derivative operators are UV-benign, eliminating
  Ostrogradsky instabilities.
key_results:
  - Finds non-Gaussian UV fixed point for gravity + scalaron via FRG
  - Demonstrates BRST-based anomaly freedom and unitarity of gauge sector
  - Proves absence of ghosts in higher-derivative curvature terms
read_link: "/static/docs/pdfs/rft-11-2.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2011.2.docx"
supplementary:
  - label: "RG-flow figures"
    file:  "rft11_2_rgplots.zip"
citation: |
  @article{Fitzpatrick2025RFT11_2,
    author  = {Fitzpatrick, I.},
    title   = {FRG and BRST Consistency of the Scalaron–Twistor Theory},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 11.2}
  }
featured: true

# ---------- RFT 11.3 ----------
slug: rft-11-3
title: "Open Phenomenology Toolkit & Validation Plan"
version: "11.3"
date: 2025-04-05
tags: ["Phenomenology", "Toolkit"]
abstract: >
  Presents a fully open-source Python/Jupyter toolkit that reproduces
  all scalaron–twistor phenomenology tracks—black-hole Page curves,
  GW echoes, primordial spectra, and high-energy transients—so the
  community can falsify or refine RFT predictions collaboratively.
key_results:
  - Ships modular notebooks for BH evaporation, GW waveforms & CMB spectra
  - Provides benchmark scenarios A–C for parameter stress-tests
  - Code released under a permissive MIT/GPL license to ensure full reproducibility
read_link: "/static/docs/pdfs/rft-11-3.pdf"
github_link: "https://github.com/rft-cosmology/RFT%2011.3"
featured: true

# ---------- RFT 11.4 ----------
slug: rft-11-4
title: "Precision Cosmology Forecasts for Scalaron–Twistor RFT"
version: "11.4"
date: 2025-04-10
tags: ["CMB", "Observational-Tests"]
abstract: >
  Maps the next-decade CMB and large-scale-structure measurements to
  concrete RFT predictions—Starobinsky-like n_s, r≈0.005, low-ℓ power
  suppression from a pre-inflationary bounce—and quantifies how
  experiments such as LiteBIRD and CMB-S4 can confirm or rule them
  at >5σ.
key_results:
  - Predicts n_s = 0.960–0.967; CMB-S4 can halve current error bars
  - Forecasts r ≈ 0.005, detectable by LiteBIRD/CMB-S4 at >5σ
  - Bounce scenario explains low-ℓ TT, EE deficits and parity anomalies
read_link: "/static/docs/pdfs/rft-11-4.pdf"
github_link: "https://github.com/rft-cosmology/RFT%2011.4"
featured: true

# ---------- RFT 11.5 ----------
slug: rft-11-5
title: "Gauge-Fixed Scalaron–Twistor Action & Feynman Rules"
version: "11.5"
date: 2025-04-15
tags: ["Field-Theory", "BRST"]
abstract: >
  Derives the completely gauge-fixed action for the scalaron–twistor
  unified theory, lists all propagators and vertices, and verifies
  1-loop renormalizability and BRST invariance, furnishing a ready
  reference for perturbative calculations.
key_results:
  - Provides closed-form Feynman rules for gravity, gauge and scalar sectors
  - Demonstrates BRST symmetry preserves gauge independence of S-matrix
  - SymPy notebook cross-checks sample scattering amplitudes for consistency
read_link: "/static/docs/pdfs/rft-11-5.pdf"
github_link: "https://github.com/rft-cosmology/RFT%2011.5"
featured: true

# ---------- RFT 11.6 ----------
slug: rft-11-6
title: "Scalaron–Twistor Planck-Scale Quantum Gravity"
version: "11.6"
date: 2025-04-22
tags: ["Quantum-Gravity", "Planck-Scale"]
abstract: >
  Extends the scalaron–twistor unified theory up to ~10¹⁹ GeV,
  demonstrating a UV-complete, asymptotically-safe quantum-gravity
  framework that remains predictive and falsifiable.
key_results:
  - Shows gravity + scalaron flows to a non-trivial UV fixed point
  - Predicts reflective black-hole echoes (ms delays) testable by LIGO
  - Provides holographic twistor mechanism that resolves singularities
read_link: "/static/docs/pdfs/rft-11-6.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2011.6.docx"
supplementary:  []
citation: |
  @article{Fitzpatrick2025RFT116,
    author  = {Fitzpatrick, I.},
    title   = {Scalaron–Twistor Planck-Scale Quantum Gravity},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 11.6}
  }
featured: true

# ---------- RFT 11.7 ----------
slug: rft-11-7
title: "Beyond-Standard-Model Phenomena in the Scalaron–Twistor Theory"
version: "11.7"
date: 2025-04-24
tags: ["BSM", "Phenomenology"]
abstract: >
  Surveys how the scalaron–twistor framework addresses dark energy,
  neutrino masses, leptogenesis and other BSM puzzles, mapping each
  prediction to near-term experiments and cosmological probes.
key_results:
  - Dynamical dark-energy sector predicts w₀ ≈ –0.98 with slight dz drift
  - Links baryogenesis to see-saw Majorana neutrinos; 0νββ decay expected
  - Forecasts GUT-scale cosmic-string GW background detectable by PTAs
read_link: "/static/docs/pdfs/rft-11-7.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2011.7.docx"
supplementary:  []
citation: |
  @article{Fitzpatrick2025RFT117,
    author  = {Fitzpatrick, I.},
    title   = {Beyond-Standard-Model Phenomena in the Scalaron–Twistor Theory},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 11.7}
  }
featured: true

# ---------- RFT 11.8 ----------
slug: rft-11-8
title: "Rigorous Formalization of the Scalaron–Twistor Unified Theory"
version: "11.8"
date: 2025-04-26
tags: ["Foundations", "Formalism"]
abstract: >
  Presents a theorem-proof treatment of the scalaron–twistor action,
  establishing mathematical consistency, BRST-safe quantization, and
  low-energy reduction to GR + Standard Model.
key_results:
  - Proves well-posed Einstein-scalaron equations and constraint closure
  - Demonstrates absence of ghosts; singularities become smooth bounces
  - Recovers GR + SM in the IR via effective-action decoupling
read_link: "/static/docs/pdfs/rft-11-8.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2011.8.docx"
supplementary:  []
citation: |
  @article{Fitzpatrick2025RFT118,
    author  = {Fitzpatrick, I.},
    title   = {Rigorous Formalization of the Scalaron–Twistor Unified Theory},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 11.8}
  }
featured: true

# ---------- RFT 12.0 ----------
slug: rft-12-0
title: "Scalaron–Twistor Phenomenology: Three Generations & SM Fit"
version: "12.0"
date: 2025-04-05
tags: ["Phenomenology", "Mass-Hierarchy", "Gauge-Unification"]
abstract: >
  Extends the unified scalaron–twistor framework to particle physics and
  cosmology. A Chern-index-3 topology in the scalaron–twistor bundle naturally
  yields exactly three fermion generations, while overlap integrals between
  localized twistor modes and the scalaron background reproduce the observed
  fermion-mass hierarchy and CKM/PMNS mixing. The same field content matches
  Standard-Model parameters, achieves gauge-coupling unification near
  10¹⁶ GeV, and remains consistent with Planck-2025 cosmology.
key_results:
  - Derives three generations from a scalaron–twistor Chern index 3
  - Fits charged-lepton and quark masses via wave-function overlaps
  - Preserves SM couplings, inflation observables, and dark-energy equation-of-state
read_link: "/static/docs/pdfs/rft-12-0.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2012.0.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT12,
    author  = {Fitzpatrick, I.},
    title   = {Scalaron–Twistor Phenomenology: Three Generations & SM Fit},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 12.0}
  }
featured: true

# ---------- RFT 12.1.1 ----------
slug: rft-12-1-1
title: "Explicit Numerical Verification of Anomaly Cancellation"
version: "12.1.1"
date: 2025-04-12
tags: ["Consistency", "Anomaly-Check"]
abstract: >
  Technical supplement to RFT 12.x that numerically confirms the
  anomaly-free and Witten-safe status of one Standard-Model generation within
  the scalaron–twistor setup. Provides step-by-step sums for all chiral gauge,
  mixed gravitational, and cubic hyper-charge anomalies and supplies a stable
  fermion-representation table for automated consistency tests.
key_results:
  - Cancels $[SU(2)]^{2}U(1)$, $[SU(3)]^{2}U(1)$, $[U(1)]^{3}$ and grav-$U(1)$ anomalies
  - Confirms no global $SU(2)$ (Witten) anomaly via even doublet count
  - Ships table & code for reproducible anomaly checks in downstream builds
read_link: "/static/docs/pdfs/rft-12-1-1.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2012.1.1.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT12_1_1,
    author  = {Fitzpatrick, I.},
    title   = {Explicit Numerical Verification of Anomaly Cancellation},
    journal = {RFT Cosmology Supplement},
    year    = {2025},
    note    = {Version 12.1.1}
  }
featured: true

# ---------- RFT 11.9 ----------
slug: rft-11-9
title: "Emergence & Decoherence in the Scalaron–Twistor Framework"
version: "11.9"
date: 2025-03-28
tags: ["Foundations", "Decoherence"]
abstract: >
  Develops the philosophical and formal underpinnings of how classical
  spacetime, locality, and apparent probabilistic outcomes emerge from an
  underlying deterministic, unitary scalaron–twistor theory. Interprets the
  decoherence term Γ₍decoh₎ as an effective coarse-graining over inaccessible
  degrees of freedom and shows how macroscopic indeterminism arises without
  violating fundamental unitarity.
key_results:
  - Explains classical spacetime emergence via large-scale decoherence
  - Interprets Γ₍decoh₎ as environmental tracing, reconciling determinism with QM statistics
  - Positions RFT as unitary yet compatible with observed macroscopic randomness
read_link: "/static/docs/pdfs/rft-11-9.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2011.9.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT11_9,
    author  = {Fitzpatrick, I.},
    title   = {Emergence & Decoherence in the Scalaron–Twistor Framework},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 11.9}
  }
featured: true

# ---------- RFT 12.1 ----------
slug: rft-12-1
title: "Renormalization-Group & Gauge-Sector Foundations"
version: "12.1"
date: 2025-04-04
tags: ["FRG", "Gauge"]
abstract: >
  Computes one-loop functional-RG flow for gravity + scalaron and
  shows how an SU(2) gauge field emerges from twistor-scalar dynamics.
key_results:
  - Derives coupled β-functions for {G, Λ, α, λ} and finds a UV non-Gaussian fixed point with two relevant directions
  - Demonstrates SU(2) gauge symmetry arises naturally from a triplet scalar in the twistor formalism
  - Confirms matter couplings are irrelevant at the fixed point, tightening RFT predictivity
read_link: "/static/docs/pdfs/rft-12-1.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2012.1.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT12_1,
    author  = {Fitzpatrick, I.},
    title   = {Renormalization-Group \& Gauge-Sector Foundations},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 12.1}
  }
featured: true

# ---------- RFT 12.2 ----------
slug: rft-12-2
title: "Deeper Structural Validation & Foundational Extensions"
version: "12.2"
date: 2025-04-12
tags: ["Foundations", "SUSY"]
abstract: >
  Extends the scalaron–twistor framework with an N = 1 supergravity
  embedding, lattice-twistor discretization and full SM gauge emergence.
key_results:
  - Embeds RFT in minimal supergravity, outlining supertwistor multiplets for the unified field
  - Shows SU(3)_C × SU(2)_L × U(1)_Y emerge from internal twistor symmetries, reproducing SM breaking patterns
  - Presents a roadmap to non-perturbative unitarity via FRG, bootstrap and positivity constraints
read_link: "/static/docs/pdfs/rft-12-2.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2012.2.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT12_2,
    author  = {Fitzpatrick, I.},
    title   = {Deeper Structural Validation \& Foundational Extensions},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 12.2}
  }
featured: true

# ---------- RFT 12.3 ----------
slug: rft-12-3
title: "UV Fixed-Point Proof & Electroweak Lattice Test"
version: "12.3"
date: 2025-04-24
tags: ["Asymptotic-Safety", "Lattice"]
abstract: >
  Establishes an asymptotically safe UV fixed point for gravity–scalar–gauge
  couplings and confirms electroweak mass relations in a 2×2 twistor lattice.
key_results:
  - Finds a non-trivial UV fixed point with two relevant directions and suppressed U(1)_Y Landau pole
  - Twistor-lattice simulation reproduces M_W/M_Z ≈ 0.88 and correct Weinberg angle to <0.1 %
  - Predicts near-unified gauge couplings at ≈10¹⁶ GeV within the RFT flow
read_link: "/static/docs/pdfs/rft-12-3.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2012.3.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT12_3,
    author  = {Fitzpatrick, I.},
    title   = {UV Fixed-Point Proof \& Electroweak Lattice Test},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 12.3}
  }
featured: true

# ---------- RFT 12.4 ----------
slug: rft-12-4
title: "Asymptotic-Safety & Gauge-Coupling Unification in RFT"
version: "12.4"
date: 2025-04-15           # adjust to repo commit date if different
tags: ["Asymptotic-Safety", "Gauge-Unification", "Twistor-Lattice"]
abstract: >
  Extends the FRG analysis to two loops, confirming a stable UV fixed
  point that keeps gravity and the scalaron finite, shows quantum gravity
  drives all three SM gauge couplings to a common safe point near
  10¹⁶ GeV, and validates the full SU(3)×SU(2)×U(1) twistor-lattice
  implementation on a 4 × 4 grid.
key_results:
  - Two-loop FRG finds UV fixed point with only two relevant directions
  - Gravity renders U(1)₍Y₎ asymptotically safe and unifies gauge couplings at ~10¹⁶ GeV
  - 4×4 twistor lattice reproduces massless gluons, chiral zero-modes, and correct EWSB pattern
read_link: "/static/docs/pdfs/rft-12-4.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2012.4.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT12_4,
    author  = {Fitzpatrick, I.},
    title   = {Asymptotic-Safety \& Gauge-Coupling Unification in RFT},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 12.4}
  }
featured: true

# ---------- RFT 12.5 ----------
slug: rft-12-5
title: "Scalaron as an SU(2)ₗ Doublet: Electroweak Breaking in RFT"
version: "12.5"
date: 2025-04-20
tags: ["Electroweak", "Scalaron", "Higgs-Mechanism"]
abstract: >
  Demonstrates that the scalaron naturally forms an SU(2)ₗ doublet,
  triggering electroweak symmetry breaking with custodial ρ = 1,
  reproducing M_W, M_Z and sin²θ_W, and generating realistic Yukawa
  hierarchies from geometric fermion-scalaron overlaps.
key_results:
  - Scalaron doublet VEV yields correct W/Z masses and preserves ρ ≈ 1
  - Localized fermion overlaps explain SM Yukawa and CKM hierarchies geometrically
  - Lattice simulation confirms massless photon and stable 246 GeV VEV across sites
read_link: "/static/docs/pdfs/rft-12-5.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2012.5.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT12_5,
    author  = {Fitzpatrick, I.},
    title   = {Scalaron as an SU(2)\_L Doublet: Electroweak Breaking in RFT},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 12.5}
  }
featured: true

# ---------- RFT 12.6.1 ----------
slug: rft-12-6-1
title: "Loop-Cancellation Mechanism for the Cosmological Constant"
version: "12.6.1"
date: 2025-04-26
tags: ["Cosmological-Constant", "Quantum-Loops", "Dark-Energy"]
abstract: >
  Resolves the vacuum-energy problem by showing how SUSY partner loops
  and discrete twistor-flux tuning cancel TeV-scale contributions,
  stabilising Λ_eff at ~10⁻⁴⁷ GeV⁴ and paving the way for RFT 12.7's
  dark-energy phenomenology.
key_results:
  - SUSY + twistor-flux cancels 1-loop vacuum energy to observed Λ_eff scale
  - Quantifies required superpartner mass splitting Δm ≈ 10⁻⁵³ GeV and shows flux quantization enforces it
  - Integrates cancellation framework into roadmap for dark-energy tests in RFT 12.7
read_link: "/static/docs/pdfs/rft-12-6-1.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2012.6.1.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT12_6_1,
    author  = {Fitzpatrick, I.},
    title   = {Loop-Cancellation Mechanism for the Cosmological Constant},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 12.6.1}
  }
featured: true

# ---------- RFT 12.6 ----------
slug: rft-12-6
title: "Twistor 4-Form Flux & FRG Solution to Λ"
version: "12.6"
date: 2025-04-15
tags: ["Cosmology", "Cosmological-Constant"]
abstract: >
  Shows how a single quantized 4-form threading twistor space—and an
  asymptotically-safe FRG trajectory—drive the effective cosmological constant
  down to Λ_eff ≈ (2 × 10⁻³ eV)⁴ without fine-tuning. Also proves the tiny Λ
  term leaves the electroweak vacuum stable.
key_results:
  - Derives Λ_eff from one 4-form flux on CP³ fiber; value matches Planck 2024 data
  - Independently recovers the same Λ via 2-loop FRG running to k → 0
  - Confirms Higgs/scalaron potential remains absolutely stable with Λ_eff included
read_link: "/static/docs/pdfs/rft-12-6.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2012.6.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT12_6,
    author  = {Fitzpatrick, I.},
    title   = {Twistor 4-Form Flux and FRG Solution to the Cosmological Constant},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 12.6}
  }
featured: true

# ---------- RFT 12.7 ----------
slug: rft-12-7
title: "Quantum Unitarity Audit of the Scalaron–Graviton Sector"
version: "12.7"
date: 2025-04-18
tags: ["Quantum-Consistency", "Unitarity"]
abstract: >
  Delivers a four-level unitarity proof for RFT: ghost-free tree
  amplitudes, conformal-bootstrap positivity, twistor-lattice S-matrix tests,
  and SM partial-wave safety. Establishes RFT 12.x as a fully unitary quantum
  gravity + matter theory.
key_results:
  - Finds only physical poles in all 2→2 graviton/scalaron amplitudes; optical theorem holds
  - Bootstrap shows all OPE coefficients positive and Δ ≥ 1 bounds respected
  - Lattice twistor simulation confirms reflection positivity; no negative-norm states detected
read_link: "/static/docs/pdfs/rft-12-7.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2012.7.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT12_7,
    author  = {Fitzpatrick, I.},
    title   = {Quantum Unitarity Audit of the Scalaron–Graviton Sector},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 12.7}
  }
featured: true

# ---------- RFT 12.8 ----------
slug: rft-12-8
title: "Standard-Model Precision with Scalaron Corrections"
version: "12.8"
date: 2025-04-22
tags: ["Phenomenology", "SM-Tests"]
abstract: >
  Benchmarks RFT against PDG 2024: particle masses, gauge couplings,
  decay widths and collider cross-sections all match experiment to
  < 0.1 %, with Planck-suppressed scalaron effects proving negligible at
  LHC energies.
key_results:
  - Reproduces W, Z, H, t masses and g₁–g₃ couplings within 0.1 % of PDG
  - Shows scalaron-Higgs mixing ≤10⁻¹⁶ ⇒ no change in H→γγ or t→bW widths
  - Predicts collider cross-sections σ(φ-mediated) ≈ 10⁻⁶⁸ pb → safely undetectable
read_link: "/static/docs/pdfs/rft-12-8.pdf"
github_link: "https://github.com/ifitzpat88/rft-cosmology/blob/main/Documents/RFT%2012.8.docx"
supplementary: []
citation: |
  @article{Fitzpatrick2025RFT12_8,
    author  = {Fitzpatrick, I.},
    title   = {Standard-Model Precision with Scalaron Corrections},
    journal = {RFT Cosmology Preprint},
    year    = {2025},
    note    = {Version 12.8}
  }
featured: true

# ---------- RFT 12.9 ----------
slug: rft-12-9
title: "Late-Universe Predictions & Experimental Roadmap"
version: "12.9"
date: 2025-04-20          # adjust if you prefer the exact commit date
tags: ["Cosmology", "Dark-Energy", "Forecasts"]
abstract: >
  Synthesises CMB, BAO and growth-rate tests for RFT; refines scalaron
  dark-energy evolution (w₀ ≈ –0.995, wₐ ≈ +0.05) and lays out a
  2025-2050 observational roadmap—LIGO-O5 echoes, Euclid lensing,
  CMB-S4 B-modes—to confirm or falsify the theory.
key_results:
  - Matches Planck CMB (nₛ≈0.965, r≈0.004) and DESI BAO at <1 %
  - Predicts GW echo delays ≈ 0.1 s at 1 % amplitude for 30 M⊙ BH mergers
  - Dark-energy thawing: w₀, wₐ values testable by Euclid + LSST
read_link: "/static/docs/pdfs/rft-12-9.pdf"
github_link: "https://github.com/rft-cosmology/RFT%2012.9"
featured: true

# ---------- RFT 13.0 ----------
slug: rft-13-0
title: "Unified Scalaron–Twistor Framework (Full ToE Draft)"
version: "13.0"
date: 2025-04-26
tags: ["Unification", "Quantum-Gravity", "Roadmap"]
abstract: >
  Finalises the gauge-fixed action unifying gravity, the Standard Model
  and cosmology in a renormalisable scalaron–twistor field theory; shows
  <0.1 % agreement with electroweak precision data and outlines
  multi-decade tests from HL-LHC to Einstein Telescope.
key_results:
  - Closed-form BRST-invariant action; 1-loop renormalisable
  - Reproduces W, Z, H, t masses within 0.1 % of experiment
  - Seesaw mechanism yields Σ mν ≈ 0.06 eV; predicts 0νββ at 10 meV
read_link: "/static/docs/pdfs/rft-13-0.pdf"
github_link: "https://github.com/rft-cosmology/RFT%2013.0"
featured: true
"""

# Run the script
if __name__ == "__main__":
    process_yaml()