#!/usr/bin/env python3
"""Ensure all section headings in index.qmd have unique anchors."""
from __future__ import annotations

import re
import sys
from pathlib import Path

ROOT = Path(__file__).resolve().parents[1]
INDEX_FILE = ROOT / "rft-vol2-arxiv" / "index.qmd"

HEADING_PATTERN = re.compile(r"^(?P<head>##+)\s+(?P<title>.*?)(?:\s*\{#(?P<slug>[^}]+)\})?\s*$")


def slugify(text: str) -> str:
    text = text.lower()
    text = re.sub(r"[^a-z0-9]+", "-", text)
    return text.strip("-") or "section"


def main() -> int:
    if not INDEX_FILE.exists():
        print("index.qmd not found", file=sys.stderr)
        return 1

    lines = INDEX_FILE.read_text(encoding="utf-8").splitlines()
    seen: dict[str, int] = {}
    changed = False

    for idx, line in enumerate(lines):
        match = HEADING_PATTERN.match(line)
        if not match:
            continue
        head, title, slug = match.group("head"), match.group("title"), match.group("slug")
        if slug is None:
            base = slugify(title)
            slug = base
            while slug in seen:
                seen[slug] += 1
                slug = f"{base}-{seen[slug]}"
            seen[slug] = 1
            lines[idx] = f"{head} {title} {{#{slug}}}"
            changed = True
        else:
            base = slug
            if base in seen:
                seen[base] += 1
                new_slug = f"{base}-{seen[base]}"
                lines[idx] = f"{head} {title} {{#{new_slug}}}"
                slug = new_slug
                changed = True
            else:
                seen[base] = 1
                slug = base
        # ensure slug recorded
        if slug not in seen:
            seen[slug] = 1

    if changed:
        INDEX_FILE.write_text("\n".join(lines) + "\n", encoding="utf-8")
        print("[lint_anchors] Applied anchor fixes to index.qmd")
    else:
        print("[lint_anchors] No anchor fixes required")
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
