#!/usr/bin/env node

/**
 * Lint script for validating papers.json data integrity
 * Checks for:
 * - No duplicate IDs
 * - All public papers have pubDate <= today
 * - Required fields are present
 * - Dates are in valid ISO format
 * - Param-sync banner date is up to date
 */

const fs = require('fs');
const path = require('path');

const PAPERS_FILE = path.join(__dirname, '..', 'app', 'static', 'data', 'papers.json');
const BASE_TEMPLATE = path.join(__dirname, '..', 'app', 'templates', 'base.html');
const CONTEXT_PROCESSOR = path.join(__dirname, '..', 'app', 'context_processors.py');

let errors = [];
let warnings = [];

function logError(message) {
    errors.push(`❌ ERROR: ${message}`);
}

function logWarning(message) {
    warnings.push(`⚠️  WARNING: ${message}`);
}

function validatePapers() {
    console.log('🔍 Validating papers.json...\n');
    
    // Check if papers.json exists
    if (!fs.existsSync(PAPERS_FILE)) {
        logError('papers.json not found at expected location');
        return false;
    }
    
    let papers;
    try {
        const content = fs.readFileSync(PAPERS_FILE, 'utf8');
        papers = JSON.parse(content);
    } catch (e) {
        logError(`Failed to parse papers.json: ${e.message}`);
        return false;
    }
    
    // Check if papers is an array
    if (!Array.isArray(papers)) {
        logError('papers.json must contain an array of papers');
        return false;
    }
    
    // Track IDs for duplicate check
    const ids = new Set();
    const today = new Date().toISOString().split('T')[0];
    let latestDate = null;
    
    papers.forEach((paper, index) => {
        const paperRef = `Paper at index ${index}`;
        
        // Check required fields
        const requiredFields = ['id', 'title', 'series', 'status', 'pubDate'];
        requiredFields.forEach(field => {
            if (!paper[field]) {
                logError(`${paperRef}: Missing required field '${field}'`);
            }
        });
        
        // Check for duplicate IDs
        if (paper.id) {
            if (ids.has(paper.id)) {
                logError(`Duplicate ID found: '${paper.id}'`);
            }
            ids.add(paper.id);
        }
        
        // Validate status
        if (paper.status && !['public', 'draft'].includes(paper.status)) {
            logError(`${paperRef} (${paper.id}): Invalid status '${paper.status}'. Must be 'public' or 'draft'`);
        }
        
        // Validate pubDate format
        if (paper.pubDate) {
            const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
            if (!dateRegex.test(paper.pubDate)) {
                logError(`${paperRef} (${paper.id}): Invalid date format '${paper.pubDate}'. Must be YYYY-MM-DD`);
            } else {
                // Check if public papers have future dates
                if (paper.status === 'public' && paper.pubDate > today) {
                    logWarning(`${paperRef} (${paper.id}): Public paper has future publication date '${paper.pubDate}'`);
                }
                
                // Track latest date
                if (paper.status === 'public' && (!latestDate || paper.pubDate > latestDate)) {
                    latestDate = paper.pubDate;
                }
            }
        }
        
        // Validate series
        if (paper.series) {
            const seriesNum = parseFloat(paper.series);
            if (isNaN(seriesNum) || seriesNum < 0) {
                logError(`${paperRef} (${paper.id}): Invalid series number '${paper.series}'`);
            }
        }
        
        // Check abstract length
        if (paper.abstract && paper.abstract.length > 500) {
            logWarning(`${paperRef} (${paper.id}): Abstract is very long (${paper.abstract.length} chars). Consider shortening.`);
        }
        
        // Check filename
        if (paper.filename) {
            if (!paper.filename.endsWith('.pdf') && !paper.filename.endsWith('.docx')) {
                logWarning(`${paperRef} (${paper.id}): Filename '${paper.filename}' should end with .pdf or .docx`);
            }
        }
    });
    
    // Check param-sync banner
    console.log('\n🔍 Checking param-sync banner...\n');
    
    if (fs.existsSync(BASE_TEMPLATE)) {
        const baseContent = fs.readFileSync(BASE_TEMPLATE, 'utf8');
        const paramSyncMatch = baseContent.match(/Params synced:.*?{{.*?latest_pub_date.*?}}/);
        
        if (paramSyncMatch) {
            console.log('✅ Param-sync banner correctly uses dynamic latest_pub_date variable');
        } else {
            const hardcodedMatch = baseContent.match(/Params synced:.*?(\d{4}[-.\d]*)/);
            if (hardcodedMatch) {
                logError(`Param-sync banner has hardcoded date: '${hardcodedMatch[1]}'. Should use {{ latest_pub_date }}`);
            }
        }
    }
    
    // Check context processor
    if (fs.existsSync(CONTEXT_PROCESSOR)) {
        const contextContent = fs.readFileSync(CONTEXT_PROCESSOR, 'utf8');
        if (contextContent.includes('latest_pub_date')) {
            console.log('✅ Context processor provides latest_pub_date');
        } else {
            logWarning('Context processor may not be providing latest_pub_date');
        }
    }
    
    // Summary
    console.log('\n📊 Summary:');
    console.log(`   Papers found: ${papers.length}`);
    console.log(`   Public papers: ${papers.filter(p => p.status === 'public').length}`);
    console.log(`   Draft papers: ${papers.filter(p => p.status === 'draft').length}`);
    if (latestDate) {
        console.log(`   Latest publication date: ${latestDate}`);
    }
    
    return true;
}

// Main execution
console.log('🚀 RFT Papers Linter\n');

validatePapers();

// Print results
if (errors.length > 0) {
    console.log('\n❌ Errors found:\n');
    errors.forEach(err => console.log(err));
}

if (warnings.length > 0) {
    console.log('\n⚠️  Warnings:\n');
    warnings.forEach(warn => console.log(warn));
}

if (errors.length === 0 && warnings.length === 0) {
    console.log('\n✅ All checks passed!');
}

// Exit with error code if errors found
if (errors.length > 0) {
    console.log('\n❌ Lint failed with errors');
    process.exit(1);
} else {
    console.log('\n✅ Lint completed successfully');
    process.exit(0);
}