#!/usr/bin/env python3
"""
Convert literature cluster lensing data (CSV) to ClusterProfile JSON schema.

Usage:
    python3 scripts/lensing_from_csv.py data/Abell1689.csv CLASH 0.183 0.8 > cases/lensing/Abell1689.json

Input CSV format (no header):
    R_kpc, DeltaSigma_Msun_pc2, sigma_Msun_pc2

Output: ClusterProfile.schema.json compliant JSON

Author: RFT Cosmology Project
Date: 2025-11-10
"""

import json
import sys
import numpy as np
from pathlib import Path


def main():
    if len(sys.argv) != 5:
        print("Usage: lensing_from_csv.py <csv_file> <survey> <z_lens> <z_source>", file=sys.stderr)
        print("", file=sys.stderr)
        print("Example:", file=sys.stderr)
        print("  python3 scripts/lensing_from_csv.py data/Abell1689.csv CLASH 0.183 0.8 > cases/lensing/Abell1689.json", file=sys.stderr)
        sys.exit(1)

    csv_file = sys.argv[1]
    survey = sys.argv[2]
    z_lens = float(sys.argv[3])
    z_source = float(sys.argv[4])

    # Load CSV (R, DeltaSigma, sigma)
    try:
        R, DeltaSigma, sigma = np.loadtxt(csv_file, delimiter=",", unpack=True)
    except Exception as e:
        print(f"ERROR loading {csv_file}: {e}", file=sys.stderr)
        print("Expected CSV format: R_kpc, DeltaSigma_Msun_pc2, sigma_Msun_pc2", file=sys.stderr)
        sys.exit(1)

    # Extract cluster name from filename
    cluster_name = Path(csv_file).stem

    # Build JSON output
    output = {
        "name": cluster_name,
        "meta": {
            "survey": survey,
            "data_source": f"Converted from {Path(csv_file).name}",
            "units": {
                "R": "kpc",
                "DeltaSigma": "Msun_per_pc2",
                "sigma": "Msun_per_pc2"
            },
            "cosmo_input": {
                "H0": 70.0,
                "Omega_m": 0.3,
                "Omega_Lambda": 0.7
            }
        },
        "lens": {
            "z": z_lens,
            "name": cluster_name
        },
        "sources": {
            "z_eff": z_source,
            "description": "Effective source redshift"
        },
        "profile": [
            {
                "R_kpc": float(r),
                "DeltaSigma_Msun_pc2": float(ds),
                "DeltaSigma_err_Msun_pc2": float(s)
            }
            for r, ds, s in zip(R, DeltaSigma, sigma)
        ]
    }

    # Write to stdout
    print(json.dumps(output, indent=2))


if __name__ == "__main__":
    main()
