#!/usr/bin/env python3
"""
C11: Grid search for mixed kernel (Gaussian + Lorentzian).

Minimal grid (12 configs):
w ∈ {0.25, 0.50, 0.75}
sigma_g ∈ {0.35, 0.45}
sigma_l ∈ {0.60, 0.80}
asym = 0.0 (no asymmetry initially)

Total: 3 × 2 × 2 = 12 configs
"""
import json
import subprocess
import sys
from pathlib import Path
from itertools import product

# Base config: C10.2 frozen with mixed kernel params
BASE_CONFIG = "config/global_c10p_frozen.json"
MANIFEST = "cases/SP99-TRAIN.manifest.txt"

# Grid parameters
W_GRID = [0.25, 0.50, 0.75]
SIGMA_G_GRID = [0.35, 0.45]
SIGMA_L_GRID = [0.60, 0.80]
ASYM = 0.0  # No asymmetry for minimal grid

def main():
    # Load base config
    with open(BASE_CONFIG) as f:
        base = json.load(f)

    # Ensure output directory exists
    Path("config").mkdir(exist_ok=True)

    results = []
    total_configs = len(W_GRID) * len(SIGMA_G_GRID) * len(SIGMA_L_GRID)
    config_num = 1

    print(f"C11 Grid Search: {total_configs} configs")
    print(f"w ∈ {W_GRID}")
    print(f"sigma_g ∈ {SIGMA_G_GRID}")
    print(f"sigma_l ∈ {SIGMA_L_GRID}")
    print(f"asym = {ASYM}")
    print(f"=" * 60)

    for w, sigma_g, sigma_l in product(W_GRID, SIGMA_G_GRID, SIGMA_L_GRID):
        # Create config
        cfg = json.loads(json.dumps(base))  # Deep copy

        # Set mixed kernel params
        cfg["mode_spiral"]["kernel"] = "mix"
        cfg["mode_spiral"]["w"] = w
        cfg["mode_spiral"]["sigma_g"] = sigma_g
        cfg["mode_spiral"]["sigma_l"] = sigma_l
        cfg["mode_spiral"]["asym"] = ASYM

        # Remove Gaussian sigma if present
        if "sigma_ln_r" in cfg["mode_spiral"]:
            del cfg["mode_spiral"]["sigma_ln_r"]

        # Save config
        config_name = f"config/global_c11_w{w:.2f}_sg{sigma_g:.2f}_sl{sigma_l:.2f}.json"
        with open(config_name, "w") as f:
            json.dump(cfg, f, indent=2)

        # Run benchmark
        suffix = f"C11_w{w:.2f}_sg{sigma_g:.2f}_sl{sigma_l:.2f}"
        print(f"\n[{config_num}/{total_configs}] {suffix}")
        print(f"Config: {config_name}")

        cmd = [
            "python3", "-m", "cli.rft_rc_bench",
            "--batch", MANIFEST,
            "--solver", "rft_geom",
            "--global-config", config_name,
            "--min-radius", "1.0",
            "--max-radius", "30.0",
            "--min-points", "10",
            "--max-workers", "0",
        ]

        try:
            subprocess.run(cmd, check=True, capture_output=True)
        except subprocess.CalledProcessError as e:
            print(f"✗ Benchmark failed:")
            print(e.stderr.decode())
            continue

        # Aggregate
        agg_cmd = [
            "python3", "-m", "batch.aggregate",
            "--restrict-manifest", MANIFEST,
            "--pass-threshold", "20.0",
            "--suffix", suffix,
        ]

        try:
            result = subprocess.run(agg_cmd, check=True, capture_output=True, text=True)
            print(result.stdout)

            # Parse results from output
            lines = result.stdout.strip().split("\n")
            for line in lines:
                if "pass rate" in line.lower() or "median rms" in line.lower():
                    print(f"  {line.strip()}")

            results.append({
                "w": w,
                "sigma_g": sigma_g,
                "sigma_l": sigma_l,
                "config": config_name,
                "suffix": suffix,
            })
        except subprocess.CalledProcessError as e:
            print(f"✗ Aggregation failed:")
            print(e.stderr)
            continue

        config_num += 1

    # Summary
    print(f"\n{'=' * 60}")
    print(f"C11 Grid Search Complete: {len(results)}/{total_configs} configs")
    print(f"\nConfigs:")
    for r in results:
        print(f"  w={r['w']:.2f}, sigma_g={r['sigma_g']:.2f}, sigma_l={r['sigma_l']:.2f}")
        print(f"    → {r['config']}")

    print(f"\nSummary files in reports/_summary_C11_*.json")
    print(f"Find best: ls -lth reports/_summary_C11_*.json | head -5")

if __name__ == "__main__":
    main()
