#!/usr/bin/env python3
"""Build `app/data/publications.json` from Omega portal outputs.

The generator keeps all public-facing metadata in a single place so that
publishing a new Volume or minor revision is a matter of updating the
`SPECS` block and rerunning this script.

It reads the Omega portal exports under
`/home/rftuser/omega_document/admin_portal/static` to discover which
artifacts (HTML/PDF) exist, enriches each entry with word counts from the
portal metadata files, and writes a sorted JSON manifest that the Flask
application consumes.
"""

from __future__ import annotations

import json
from pathlib import Path
from typing import Any, Dict, List

BASE_DIR = Path(__file__).resolve().parents[1]
MANIFEST_PATH = BASE_DIR / "app" / "data" / "publications.json"
PORTAL_BASE_URL = "http://164.92.99.99:5003/standalone"
PORTAL_ROOT = Path("/home/rftuser/omega_document/admin_portal/static")
HTML_STANDALONE = PORTAL_ROOT / "html" / "standalone"
STANDALONE_ASSETS = PORTAL_ROOT / "standalone_assets"

# ---------------------------------------------------------------------------
# Publication descriptors — extend this dictionary when a new volume ships.
# ---------------------------------------------------------------------------
SPECS: Dict[str, Dict[str, Any]] = {
    "rft-volume1-rc11": {
        "title": "Resonant Field Theory, Volume 1: Unified Field Dynamics",
        "short_title": "RFT Volume 1",
        "subtitle": "Scalaron–Twistor Spacetime, Environmental Screening, and Observational Gates",
        "version": "v1.0-rc11",
        "status": "Preprint",
        "release_date": "2025-09-21",
        "authors": ["Ian Fitzpatrick"],
        "summary": (
            "Establishes the scalaron–twistor lattice as a unified description of gravity and gauge sectors, "
            "demonstrating emergent spacetime, environmental screening, and preregistered observational gates "
            "that match Cassini, Planck, and SPARC data."
        ),
        "highlights": [
            "Derives Einstein limits from scalaron–twistor thermodynamics while keeping spacetime discrete at the microscale.",
            "Demonstrates screened modified gravity consistent with Solar System constraints yet reproducing galaxy rotation curves and stacked weak lensing.",
            "Shows E₈→E₆×SU(3) breaking yields three fermion generations and Dirac-only neutrinos, with deterministic build harness for reproducibility."
        ],
        "tags": ["modified-gravity", "screening", "emergent-spacetime", "volume-1"],
        "page_count": 140,
        "portal_slug": "rft-volume1-rc11",
        "source_repo": "https://github.com/rft-cosmology/rft-vol1-rc11",
        "repro_repo": "https://github.com/rft-cosmology/rft-vol1-proofpack",
    },
    "rft-volume2-rc12": {
        "title": "Resonant Field Theory, Volume 2: Geometric Necessity",
        "short_title": "RFT Volume 2",
        "subtitle": "Constants from Geometry, Embedding Uniqueness, and Nuclear Closures",
        "version": "v2.0-rc12",
        "status": "Preprint",
        "release_date": "2025-09-30",
        "authors": ["Ian Fitzpatrick"],
        "summary": (
            "Derives the five fundamental constants (ℏ, α, mₑ/mₚ, c, G) as geometric consequences of the scalaron–twistor "
            "lattice, links SU(3) instantons to family replication, and presents nuclear shell closures from Platonic geometry."
        ),
        "highlights": [
            "Shows ℏ and α emerge from CP¹ Berry holonomy and dual polyhedral closure, matching CODATA at sub-ppm precision without parameter tuning.",
            "Solves a 3×3 lattice eigenproblem for the 1836:1 mass hierarchy and relates BAO, growth, and dark energy parameters to the frozen ledger.",
            "Provides falsifiable predictions and automated uncertainty budgets with reproducible build stamps and ablation tests on lattice geometry."
        ],
        "tags": ["fundamental-constants", "lattice-geometry", "volume-2"],
        "page_count": 50,
        "portal_slug": "rft-volume2-rc12",
        "source_repo": "https://github.com/rft-cosmology/rft-vol2-arxiv",
        "repro_repo": "https://github.com/rft-cosmology/rft-vol2-proofpack",
        "determinism_stamp": "/home/rftuser/rft-vol2-arxiv/rft-rc12-mini/stamps/determinism.txt",
    },
}

# ---------------------------------------------------------------------------
# Helper functions
# ---------------------------------------------------------------------------

def _portal_links(slug: str) -> Dict[str, str | None]:
    base = f"{PORTAL_BASE_URL}/{slug}"
    html_pdf = HTML_STANDALONE / f"{slug}.pdf"
    static_pdf = HTML_STANDALONE / f"{slug}.static.pdf"
    return {
        "html": base,
        "simplified": f"{base}-simple",
        "interactive": f"{base}-enhanced",
        "pdf": f"{base}.pdf" if html_pdf.exists() else None,
        "static_pdf": f"{base}.static.pdf" if static_pdf.exists() else None,
    }


def _load_metadata(slug: str) -> Dict[str, Any] | None:
    metadata_path = STANDALONE_ASSETS / slug / "metadata.json"
    if not metadata_path.exists():
        return None
    try:
        return json.loads(metadata_path.read_text(encoding="utf-8"))
    except json.JSONDecodeError:
        return None


def _default_timeline(spec: Dict[str, Any]) -> List[Dict[str, str]]:
    label = f"{spec.get('version', '').upper()} build published".strip()
    date = spec.get("release_date", "")
    return [{"label": label or "Build published", "date": date}]


def build_manifest() -> List[Dict[str, Any]]:
    manifest: List[Dict[str, Any]] = []

    for slug, spec in SPECS.items():
        portal_slug = spec.get("portal_slug", slug)
        metadata = _load_metadata(portal_slug) or {}
        word_count = metadata.get("word_count")
        updated_at = metadata.get("updated_at")

        timeline_spec = spec.get("timeline")
        if timeline_spec:
            timeline = [dict(event) for event in timeline_spec]
        else:
            timeline = _default_timeline(spec)

        if updated_at:
            updated_date = updated_at.split("T")[0]
            already_logged = any(event.get("date") == updated_date for event in timeline)
            if not already_logged:
                timeline.append({"label": "Omega portal rebuild", "date": updated_date})

        entry = {
            "slug": slug,
            "title": spec["title"],
            "short_title": spec.get("short_title"),
            "subtitle": spec.get("subtitle", ""),
            "version": spec.get("version", ""),
            "status": spec.get("status", "Draft"),
            "release_date": spec.get("release_date", ""),
            "authors": spec.get("authors", []),
            "summary": spec.get("summary", ""),
            "highlights": spec.get("highlights", []),
            "portal": _portal_links(portal_slug),
            "artifacts": {
                "determinism_stamp": spec.get("determinism_stamp"),
                "source_repo": spec.get("source_repo"),
                "repro_repo": spec.get("repro_repo"),
                "notebook_bundle": spec.get("notebook_bundle"),
            },
            "metrics": {
                "page_count": spec.get("page_count"),
                "word_count": word_count,
            },
            "tags": spec.get("tags", []),
            "timeline": timeline,
            "updated_at": updated_at,
        }

        manifest.append(entry)

    manifest.sort(key=lambda item: item.get("release_date", ""), reverse=True)
    return manifest


def main() -> None:
    manifest = build_manifest()
    MANIFEST_PATH.parent.mkdir(parents=True, exist_ok=True)
    MANIFEST_PATH.write_text(json.dumps(manifest, indent=2) + "\n", encoding="utf-8")
    print(f"Wrote {len(manifest)} publications → {MANIFEST_PATH}")


if __name__ == "__main__":
    main()
