#!/usr/bin/env python3
"""
Generate mock cluster weak lensing data for pilot testing.

Creates realistic cluster ΔΣ(R) profiles with:
- Diverse masses (10^14 to 10^15 Msun)
- Diverse redshifts (0.2 to 0.6)
- NFW-like profiles with realistic scatter
- Proper units (Msun/pc²)

Author: RFT Cosmology Project
Date: 2025-11-10
"""

import numpy as np
import json
from pathlib import Path
import sys

# Add project root to path
PROJECT_ROOT = Path(__file__).resolve().parents[1]
sys.path.insert(0, str(PROJECT_ROOT))

from baselines.nfw_lensing import nfw_params_from_M200_c, nfw_delta_sigma_analytical


def generate_mock_cluster(name, M200, c, z_lens, z_source, survey="MOCK"):
    """
    Generate mock cluster lensing profile.

    Parameters
    ----------
    name : str
        Cluster name
    M200 : float
        Virial mass in 10^14 Msun
    c : float
        Concentration parameter
    z_lens : float
        Lens redshift
    z_source : float
        Effective source redshift
    survey : str
        Survey name

    Returns
    -------
    cluster_data : dict
        ClusterProfile.schema.json compliant
    """
    # Generate radial bins (100-2000 kpc, 12 bins)
    R_kpc = np.logspace(np.log10(100), np.log10(2000), 12)

    # Compute NFW ΔΣ(R)
    r_s, rho_s = nfw_params_from_M200_c(M200, c, z_lens)
    DeltaSigma_true = nfw_delta_sigma_analytical(R_kpc, r_s, rho_s)

    # Add realistic scatter (10% at inner radii, 20% at outer)
    scatter_frac = 0.10 + 0.10 * (R_kpc / R_kpc.max())
    sigma = scatter_frac * DeltaSigma_true

    # Add Gaussian noise
    np.random.seed(hash(name) % 2**32)  # Reproducible per cluster
    noise = np.random.randn(len(R_kpc)) * sigma
    DeltaSigma_obs = DeltaSigma_true + noise

    # Build JSON
    cluster_data = {
        "name": name,
        "meta": {
            "survey": survey,
            "data_source": "Mock data for pilot testing",
            "units": {
                "R": "kpc",
                "DeltaSigma": "Msun_per_pc2",
                "sigma": "Msun_per_pc2"
            },
            "cosmo_input": {
                "H0": 70.0,
                "Omega_m": 0.3,
                "Omega_Lambda": 0.7
            },
            "true_params": {
                "M200_1e14": M200,
                "c": c,
                "comment": "True NFW parameters used for mock generation"
            }
        },
        "lens": {
            "z": z_lens,
            "name": name
        },
        "sources": {
            "z_eff": z_source,
            "description": "Effective source redshift"
        },
        "profile": [
            {
                "R_kpc": float(r),
                "DeltaSigma_Msun_pc2": float(ds),
                "DeltaSigma_err_Msun_pc2": float(s)
            }
            for r, ds, s in zip(R_kpc, DeltaSigma_obs, sigma)
        ]
    }

    return cluster_data


def main():
    """Generate diverse pilot set."""

    # Create output directory
    output_dir = Path("cases/lensing")
    output_dir.mkdir(parents=True, exist_ok=True)

    # Define pilot set: diverse in mass and redshift
    pilot_clusters = [
        # Low-z, low-mass
        {"name": "MockCL_A001", "M200": 5.0, "c": 4.5, "z_lens": 0.20, "z_source": 0.70},
        {"name": "MockCL_A002", "M200": 7.0, "c": 4.2, "z_lens": 0.22, "z_source": 0.75},

        # Low-z, high-mass
        {"name": "MockCL_B001", "M200": 12.0, "c": 3.8, "z_lens": 0.25, "z_source": 0.80},
        {"name": "MockCL_B002", "M200": 15.0, "c": 3.5, "z_lens": 0.28, "z_source": 0.85},

        # Mid-z, medium-mass
        {"name": "MockCL_C001", "M200": 8.0, "c": 4.0, "z_lens": 0.35, "z_source": 0.90},
        {"name": "MockCL_C002", "M200": 10.0, "c": 3.9, "z_lens": 0.38, "z_source": 0.95},
        {"name": "MockCL_C003", "M200": 9.0, "c": 4.1, "z_lens": 0.40, "z_source": 1.00},

        # High-z, diverse mass
        {"name": "MockCL_D001", "M200": 6.0, "c": 4.3, "z_lens": 0.50, "z_source": 1.10},
        {"name": "MockCL_D002", "M200": 11.0, "c": 3.7, "z_lens": 0.52, "z_source": 1.15},
        {"name": "MockCL_D003", "M200": 14.0, "c": 3.6, "z_lens": 0.55, "z_source": 1.20},
    ]

    print("=" * 70)
    print("MOCK LENSING DATA GENERATOR (PILOT)")
    print("=" * 70)
    print()

    # Generate each cluster
    for params in pilot_clusters:
        cluster_data = generate_mock_cluster(**params)

        # Write to JSON
        output_file = output_dir / f"{params['name']}.json"
        with open(output_file, "w") as f:
            json.dump(cluster_data, f, indent=2)

        print(f"✓ {params['name']}: M200={params['M200']:.1f}e14, z_l={params['z_lens']:.2f} → {output_file}")

    # Create pilot manifest
    manifest_file = Path("cases/lensing_PILOT.manifest.txt")
    cluster_files = sorted(output_dir.glob("MockCL_*.json"))
    with open(manifest_file, "w") as f:
        for cf in cluster_files:
            f.write(f"{cf}\n")

    print()
    print(f"✓ Created manifest: {manifest_file} ({len(cluster_files)} clusters)")
    print()
    print("=" * 70)
    print("PILOT READY")
    print("=" * 70)
    print()
    print("Next steps:")
    print("  1. Freeze config: make -f Makefile.lensing l-freeze")
    print("  2. Run pilot: python3 -m cli.lensing_bench --batch cases/lensing_PILOT.manifest.txt --solver rft_projected")
    print("  3. Run pilot: python3 -m cli.lensing_bench --batch cases/lensing_PILOT.manifest.txt --solver nfw_lensing")
    print("  4. Aggregate: python3 -m batch.aggregate_lensing --restrict-manifest cases/lensing_PILOT.manifest.txt --suffix L_PILOT")
    print("  5. Gate check: jq '.bic_comparisons' results/lensing/_summary_L_PILOT.json")
    print()


if __name__ == "__main__":
    main()
