#!/usr/bin/env python3
"""Generate uncertainty footers for RC12 figures."""
from __future__ import annotations

import csv
from pathlib import Path

ROOT = Path(__file__).resolve().parents[1]
RFT_DIR = ROOT / "rft-vol2-arxiv"
TABLE_DIR = RFT_DIR / "results" / "tables"
OUTPUT_ROOT = RFT_DIR / "results" / "generated" / "uncert_footers"

EXPECTED_COLUMNS = ["central", "stat", "sys", "discretization", "dominant", "gate"]

def format_row(row: dict[str, str]) -> list[str]:
    return [row.get(col, "") for col in EXPECTED_COLUMNS]

def main() -> int:
    OUTPUT_ROOT.mkdir(parents=True, exist_ok=True)
    for footer in OUTPUT_ROOT.glob("*.md"):
        footer.unlink()

    for csv_path in sorted(TABLE_DIR.glob("*_uncert.csv")):
        try:
            with csv_path.open("r", encoding="utf-8") as handle:
                reader = csv.DictReader(handle)
                rows = list(reader)
        except Exception as exc:
            print(f"[WARN] Unable to read {csv_path}: {exc}")
            continue

        if not rows:
            print(f"[WARN] No data in {csv_path}")
            continue

        stem = csv_path.name.replace("_uncert.csv", "")
        out_path = OUTPUT_ROOT / f"{stem}.md"
        lines = [
            "| central | stat | sys | discretization | dominant | gate |",
            "| --- | --- | --- | --- | --- | --- |",
            "| " + " | ".join(format_row(rows[0])) + " |",
        ]
        out_path.write_text("\n".join(lines) + "\n", encoding="utf-8")
    return 0

if __name__ == "__main__":
    raise SystemExit(main())
