#!/usr/bin/env python3
"""Generate the RC12 claims scoreboard include."""
from __future__ import annotations

import csv
import sys
import re
from datetime import datetime, timezone
from pathlib import Path

ROOT = Path(__file__).resolve().parents[1]
RFT_DIR = ROOT / "rft-vol2-arxiv"
CLAIMS_CSV = RFT_DIR / "results" / "tables" / "claims_status.csv"
INDEX_FILE = RFT_DIR / "index.qmd"
OUTPUT_DIR = RFT_DIR / "results" / "generated"
OUTPUT_FILE = OUTPUT_DIR / "claims_scoreboard.md"

STATUS_ICON = {
    "pass": "✅ pass",
    "fail": "❌ fail",
    "manual": "⚠️ pending",
    "pending": "⚠️ pending",
}


def _slugify(text: str) -> str:
    text = text.lower()
    text = re.sub(r"[^a-z0-9]+", "-", text)
    return text.strip("-")


def load_anchor_map() -> dict[str, str]:
    anchors: dict[str, str] = {}
    if not INDEX_FILE.exists():
        return anchors
    heading_pattern = re.compile(r"^(?P<hash>##+)[^S]*(?P<section>S[0-9]+(?:\.[0-9]+)?)\b.*\{#(?P<slug>[^}]+)\}")
    for line in INDEX_FILE.read_text(encoding="utf-8").splitlines():
        match = heading_pattern.match(line.strip())
        if match:
            anchors[match.group("section")] = match.group("slug")
    return anchors


def parse_float(value: str | None) -> float | None:
    if value is None or value.strip() == "":
        return None
    try:
        return float(value)
    except ValueError:
        return None


def render_table(rows: list[dict[str, str]], anchors: dict[str, str]) -> str:
    header = "| ID | Section | Status | Measured | Target | Tol | Δ | Units |\n" \
        "|---|---|---|---|---|---|---|---|"
    lines = [header]
    for row in rows:
        status = STATUS_ICON.get(row.get("status", "").lower(), row.get("status", "pending"))
        measured = parse_float(row.get("measured"))
        target = parse_float(row.get("target"))
        delta = "" if measured is None or target is None else f"{measured - target:.6g}"
        measured_str = row.get("measured", "") if measured is None else f"{measured:.6g}"
        section = row.get("section", "")
        anchor = anchors.get(section)
        section_cell = f"[{section}](#{anchor})" if anchor else section
        lines.append(
            "| {id} | {section} | {status} | {measured} | {target} | {tol} | {delta} | {units} |".format(
                id=row.get("id", ""),
                section=section_cell,
                status=status,
                measured=measured_str,
                target=row.get("target", ""),
                tol=row.get("tol", ""),
                delta=delta,
                units=row.get("units", ""),
            )
        )
    return "\n".join(lines)


def main() -> int:
    OUTPUT_DIR.mkdir(parents=True, exist_ok=True)
    anchors = load_anchor_map()

    if not CLAIMS_CSV.exists():
        OUTPUT_FILE.write_text(
            "_Claims scoreboard unavailable – run generate_rc12_artifacts.py to refresh._\n",
            encoding="utf-8",
        )
        return 0

    with CLAIMS_CSV.open("r", encoding="utf-8") as handle:
        reader = csv.DictReader(handle)
        rows = list(reader)

    generated = datetime.now(timezone.utc).strftime("%Y-%m-%d %H:%M:%SZ")
    pass_count = sum(1 for row in rows if row.get("status", "").lower() == "pass")
    lines = [
        "<!-- Auto-generated by gen_claims_scoreboard.py -->",
        f"_Generated {generated} · pass rate {pass_count}/{len(rows)}_",
        "",
        render_table(rows, anchors),
        "",
    ]
    OUTPUT_FILE.write_text("\n".join(lines), encoding="utf-8")
    return 0


if __name__ == "__main__":
    sys.exit(main())
