#!/usr/bin/env python3
"""
Extract C9 winner from grid search logs.
Parses Stage A (β) and Stage B (shelf) results to find optimal config.
"""
import re
import sys
from pathlib import Path

def parse_stage_a(log_text):
    """Extract best β parameters from Stage A."""
    best = None
    pattern = r'β0=(\S+)\s+β1=(\S+)\s+β2=(\S+).*?pass=(\d+)/(\d+).*?RMS=(\S+)%.*?LSB=(\d+)/(\d+)'

    for line in log_text.splitlines():
        m = re.search(pattern, line)
        if m:
            b0, b1, b2 = float(m.group(1)), float(m.group(2)), float(m.group(3))
            passes = int(m.group(4))
            n_cases = int(m.group(5))
            median_rms = float(m.group(6))
            lsb_passes = int(m.group(7))
            lsb_total = int(m.group(8))

            # Score: (passes desc, -median_rms, lsb_passes desc)
            score = (passes, -median_rms, lsb_passes)

            if best is None or score > best['score']:
                best = {
                    'score': score,
                    'beta0': b0,
                    'beta1': b1,
                    'beta2': b2,
                    'passes': passes,
                    'n_cases': n_cases,
                    'median_rms': median_rms,
                    'lsb_passes': lsb_passes,
                    'lsb_total': lsb_total,
                }

    return best

def parse_stage_b(log_text):
    """Extract best shelf parameters from Stage B."""
    best = None
    pattern = r'A_shelf=(\S+)\s+p=(\S+).*?pass=(\d+)/(\d+).*?RMS=(\S+)%.*?LSB=(\d+)/(\d+)'

    for line in log_text.splitlines():
        m = re.search(pattern, line)
        if m:
            a_shelf = float(m.group(1))
            p = float(m.group(2))
            passes = int(m.group(3))
            n_cases = int(m.group(4))
            median_rms = float(m.group(5))
            lsb_passes = int(m.group(6))
            lsb_total = int(m.group(7))

            score = (passes, -median_rms, lsb_passes)

            if best is None or score > best['score']:
                best = {
                    'score': score,
                    'A_shelf': a_shelf,
                    'p': p,
                    'passes': passes,
                    'n_cases': n_cases,
                    'median_rms': median_rms,
                    'lsb_passes': lsb_passes,
                    'lsb_total': lsb_total,
                }

    return best

def main():
    log_path = Path("c9_grid_search.log")

    if not log_path.exists():
        print(f"ERROR: {log_path} not found", file=sys.stderr)
        sys.exit(1)

    log_text = log_path.read_text()

    # Parse both stages
    beta_best = parse_stage_a(log_text)
    shelf_best = parse_stage_b(log_text)

    if not beta_best:
        print("ERROR: No Stage A results found in log", file=sys.stderr)
        sys.exit(1)

    print("=" * 60)
    print("C9 OPTIMAL CONFIGURATION")
    print("=" * 60)
    print()
    print("STAGE A: β-Mapping (Descriptor-driven A_flat)")
    print(f"  β0 = {beta_best['beta0']:.3f}")
    print(f"  β1 = {beta_best['beta1']:.3f}")
    print(f"  β2 = {beta_best['beta2']:.3f}")
    n_train = beta_best['n_cases']
    print(f"  Performance: {beta_best['passes']}/{n_train} @ 20% ({beta_best['passes']*100/n_train:.1f}%)")
    print(f"  Median RMS: {beta_best['median_rms']:.1f}%")
    print(f"  LSB: {beta_best['lsb_passes']}/{beta_best['lsb_total']}")
    print()

    if shelf_best:
        print("STAGE B: Shelf Parameters")
        print(f"  A_shelf = {shelf_best['A_shelf']:.3f}")
        print(f"  p = {shelf_best['p']:.2f}")
        n_train_b = shelf_best['n_cases']
        print(f"  Performance: {shelf_best['passes']}/{n_train_b} @ 20% ({shelf_best['passes']*100/n_train_b:.1f}%)")
        print(f"  Median RMS: {shelf_best['median_rms']:.1f}%")
        print(f"  LSB: {shelf_best['lsb_passes']}/{shelf_best['lsb_total']}")
    else:
        print("STAGE B: Not yet completed or no shelf improvement")
        print("  A_shelf = 0.0 (shelf disabled)")
        print("  p = 1.5 (default)")

    print()
    print("=" * 60)
    print("Next steps:")
    print("1. Verify these parameters in config/global_c9.json")
    print("2. Generate SHA256: sha256sum config/global_c9.json")
    print("3. Fill PRE_REGISTRATION_C9.md with predictions")
    print("4. Commit and tag: git tag c9-pre")
    print("5. Run TEST validation")
    print("=" * 60)

if __name__ == "__main__":
    main()
