#!/usr/bin/env python3
"""
Convert ablation run outputs into a compact JSON summary for the findings page.
"""

from __future__ import annotations

import json
from pathlib import Path
from typing import Dict


PROJECT_ROOT = Path(__file__).resolve().parents[1]
SOURCE_PATH = PROJECT_ROOT / "reports/ablations/ablation_results.json"
OUTPUT_PATH = PROJECT_ROOT / "data/refs/v2_ablations.json"

ABLATION_ORDER = [
    ("tail_off", "no_tail", "No Tail (A₀ = 0)"),
    ("no_g_gate", "no_g_gate", "No Acceleration Gate"),
    ("no_r_gate", "no_r_gate", "No Radial Onset"),
    ("alpha1_const_v", "alpha1_const_v", "Alpha = 1.0 (Constant-V Tail)"),
    ("gate_soft", "gate_soft", "Gate Softened (γ × 0.75)"),
]


def load_payload(path: Path) -> Dict:
    if not path.exists():
        raise SystemExit(f"Missing ablation results: {path}")
    return json.loads(path.read_text(encoding="utf-8"))


def main() -> None:
    raw = load_payload(SOURCE_PATH)
    baseline = raw["frozen_baseline"]
    baseline_rate = baseline["pass_20_rate"]
    baseline_count = baseline["pass_20_count"]
    n_cases = baseline.get("n_cases", 34)

    variants = []
    for source_key, variant_id, label in ABLATION_ORDER:
        entry = raw.get(source_key)
        if not entry:
            continue
        rate = entry["pass_20_rate"]
        delta = rate - baseline_rate
        variants.append(
            {
                "id": variant_id,
                "label": label,
                "pass_20_rate": rate,
                "pass_20_count": entry["pass_20_count"],
                "median_rms_pct": entry.get("rms_median"),
                "delta_pp": delta,
            }
        )

    payload = {
        "baseline": {
            "label": "Baseline (all physics on)",
            "pass_20_rate": baseline_rate,
            "pass_20_count": baseline_count,
            "median_rms_pct": baseline.get("rms_median"),
            "n_cases": n_cases,
        },
        "variants": variants,
    }

    OUTPUT_PATH.parent.mkdir(parents=True, exist_ok=True)
    OUTPUT_PATH.write_text(json.dumps(payload, indent=2), encoding="utf-8")
    print(f"[ablations] Wrote {OUTPUT_PATH} ({len(variants)} variants).")


if __name__ == "__main__":
    main()
