#!/usr/bin/env python3
"""
Emit LaTeX macros for frozen headline numbers.

Reads paper/build/final_numbers.json (hash-locked) and writes
paper/build/numbers.tex so the paper sources can reference a single,
authoritative set of values. This prevents hand-typing headline
percentages or p-values in the LaTeX.
"""

from __future__ import annotations

import json
from pathlib import Path
from typing import Dict, Iterable

PROJECT_ROOT = Path(__file__).resolve().parents[1]
FINAL_NUMBERS_PATH = PROJECT_ROOT / "paper" / "build" / "final_numbers.json"
OUTPUT_PATH = PROJECT_ROOT / "paper" / "build" / "numbers.tex"


def pct(value: float, decimals: int = 1) -> str:
    """Format a percentage value with escaped percent sign."""
    return f"{value:.{decimals}f}\\%"


def pp(value: float, decimals: int = 1) -> str:
    """Format a percentage-point delta without the percent sign."""
    return f"{value:.{decimals}f}"


def format_ci(ci: Iterable[float]) -> Iterable[str]:
    """Convert Wilson CI (stored as 0-1 floats) to percentage strings."""
    for bound in ci:
        yield pct(bound * 100.0)


def write_macros(macros: Dict[str, str]) -> None:
    """Write \newcommand macros to numbers.tex."""
    OUTPUT_PATH.parent.mkdir(parents=True, exist_ok=True)

    lines = [
        "% Auto-generated by scripts/emit_tex_numbers.py",
        "% Do not edit manually. Update paper/build/final_numbers.json instead.",
        "",
    ]
    for name, value in macros.items():
        lines.append(f"\\newcommand{{\\{name}}}{{{value}}}")

    OUTPUT_PATH.write_text("\n".join(lines) + "\n")


def main() -> int:
    if not FINAL_NUMBERS_PATH.exists():
        raise FileNotFoundError(f"Missing frozen numbers: {FINAL_NUMBERS_PATH}")

    with FINAL_NUMBERS_PATH.open() as fh:
        data = json.load(fh)

    metadata = data["metadata"]
    rft = data["aggregate"]["rft_v2"]
    nfw = data["aggregate"]["nfw_global"]
    mond = data["aggregate"]["mond"]
    paired_nfw = data["paired_tests"]["rft_vs_nfw"]
    paired_mond = data["paired_tests"]["rft_vs_mond"]

    ci_rft_low, ci_rft_high = format_ci(rft["wilson_ci"])
    ci_nfw_low, ci_nfw_high = format_ci(nfw["wilson_ci"])
    ci_mond_low, ci_mond_high = format_ci(mond["wilson_ci"])

    delta_nfw = rft["pass_rate"] - nfw["pass_rate"]
    delta_mond = rft["pass_rate"] - mond["pass_rate"]

    rft_fail = metadata["n_galaxies"] - rft["pass_count"]

    macros: Dict[str, str] = {
        "testCohortSize": str(metadata["n_galaxies"]),
        "frozenTag": metadata["tag"],
        "frozenCommit": metadata["commit"],
        "frozenDate": metadata["date"],
        "rftPassTwentyCount": str(rft["pass_count"]),
        "rftFailCount": str(rft_fail),
        "rftPassTwentyPct": pct(rft["pass_rate"]),
        "rftPassTwentyFrac": f"{rft['pass_count']}/{metadata['n_galaxies']}",
        "rftWilsonLow": ci_rft_low,
        "rftWilsonHigh": ci_rft_high,
        "nfwPassTwentyCount": str(nfw["pass_count"]),
        "nfwPassTwentyPct": pct(nfw["pass_rate"]),
        "nfwPassTwentyFrac": f"{nfw['pass_count']}/{metadata['n_galaxies']}",
        "nfwWilsonLow": ci_nfw_low,
        "nfwWilsonHigh": ci_nfw_high,
        "mondPassTwentyCount": str(mond["pass_count"]),
        "mondPassTwentyPct": pct(mond["pass_rate"]),
        "mondPassTwentyFrac": f"{mond['pass_count']}/{metadata['n_galaxies']}",
        "mondWilsonLow": ci_mond_low,
        "mondWilsonHigh": ci_mond_high,
        "rftVsNfwDeltaPP": pp(delta_nfw),
        "rftVsMondDeltaPP": pp(delta_mond),
        "mcnemarPvsNFW": f"{paired_nfw['mcnemar_p']:.2f}",
        "mcnemarPvsMOND": f"{paired_mond['mcnemar_p']:.3f}",
        "rftVsNfwBothPass": str(paired_nfw["contingency"]["both_pass"]),
        "rftVsNfwRftOnly": str(paired_nfw["contingency"]["rft_only"]),
        "rftVsNfwCompOnly": str(paired_nfw["contingency"]["competitor_only"]),
        "rftVsNfwBothFail": str(paired_nfw["contingency"]["both_fail"]),
        "rftVsMondBothPass": str(paired_mond["contingency"]["both_pass"]),
        "rftVsMondRftOnly": str(paired_mond["contingency"]["rft_only"]),
        "rftVsMondCompOnly": str(paired_mond["contingency"]["competitor_only"]),
        "rftVsMondBothFail": str(paired_mond["contingency"]["both_fail"]),
    }

    write_macros(macros)
    print(f"✅ Wrote {len(macros)} macros to {OUTPUT_PATH}")
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
