#!/bin/bash
# Quick demo of C7 baseline solvers vs RFT
# Run all 3 solvers on NGC 2403 and compare

set -e

echo "=========================================="
echo "Ticket C7 Demo: RFT vs MOND vs NFW"
echo "=========================================="
echo ""
echo "Running all 3 solvers on NGC 2403..."
echo ""

# RFT
echo "[1/3] RFT (geometric model)..."
python3 -m cli.rft_rc_bench \
  --case cases/NGC_2403.json \
  --solver rft_geom \
  --global-config config/global.json \
  --quiet

RFT_RMS=$(python3 -c "import json; m=json.load(open('reports/NGC_2403/metrics.json')); print(f\"{m['metrics']['rms_percent']:.2f}\")")

# MOND
echo "[2/3] MOND (standard law, a0=1.2e-10)..."
python3 -m cli.rft_rc_bench \
  --case cases/NGC_2403.json \
  --solver mond \
  --mond-law standard \
  --a0 1.2e-10 \
  --quiet

MOND_RMS=$(python3 -c "import json; m=json.load(open('reports/NGC_2403/metrics.json')); print(f\"{m['metrics']['rms_percent']:.2f}\")")

# NFW
echo "[3/3] NFW (dark matter halo fit)..."
python3 -m cli.rft_rc_bench \
  --case cases/NGC_2403.json \
  --solver nfw_fit \
  --nfw-bounds "rho_s=1e6,1e9;r_s=0.1,100" \
  --nfw-init "rho_s=1e8;r_s=10" \
  --quiet

NFW_RMS=$(python3 -c "import json; m=json.load(open('reports/NGC_2403/metrics.json')); print(f\"{m['metrics']['rms_percent']:.2f}\")")

echo ""
echo "=========================================="
echo "RESULTS: NGC 2403 Rotation Curve"
echo "=========================================="
echo ""
printf "%-15s %10s %10s\n" "Solver" "RMS%" "Status"
echo "------------------------------------------"
printf "%-15s %10s %10s\n" "RFT" "$RFT_RMS" "$([ ${RFT_RMS%.*} -lt 10 ] && echo '✅ PASS' || echo '❌ FAIL')"
printf "%-15s %10s %10s\n" "MOND" "$MOND_RMS" "$([ ${MOND_RMS%.*} -lt 10 ] && echo '✅ PASS' || echo '❌ FAIL')"
printf "%-15s %10s %10s\n" "NFW" "$NFW_RMS" "$([ ${NFW_RMS%.*} -lt 10 ] && echo '✅ PASS' || echo '❌ FAIL')"
echo ""
echo "Threshold: 10% RMS"
echo ""

# Determine winner
if (( $(echo "$RFT_RMS < $MOND_RMS" | bc -l) )) && (( $(echo "$RFT_RMS < $NFW_RMS" | bc -l) )); then
    echo "🏆 Winner: RFT"
elif (( $(echo "$MOND_RMS < $RFT_RMS" | bc -l) )) && (( $(echo "$MOND_RMS < $NFW_RMS" | bc -l) )); then
    echo "🏆 Winner: MOND"
else
    echo "🏆 Winner: NFW"
fi

echo ""
echo "See plots: reports/NGC_2403/curve.png"
echo "See metrics: reports/NGC_2403/metrics.json"
