#!/usr/bin/env python3

import os
from reportlab.lib.pagesizes import letter
from reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.units import inch
from reportlab.lib.enums import TA_CENTER

# Directory containing the papers
PAPERS_DIR = "/home/rftuser/rft-cosmology-site/static/papers/15x"

# Paper metadata
papers = {
    "RFT 15.1 – Twistor Quantization & SM Emergence": {
        "title": "RFT 15.1: Twistor Quantization & Standard-Model Emergence",
        "abstract": "This paper explores the emergence of the Standard Model from twistor quantization within the Resonant Field Theory framework. We demonstrate how the gauge groups, fermion families, and Higgs mechanism arise naturally from the geometric structure of twistor space.",
        "note": "This is a placeholder PDF. The actual paper content is available in DOCX format."
    },
    "RFT 15.2 – Matrix–Scalaron Quantum Gravity": {
        "title": "RFT 15.2: Matrix–Scalaron Quantum Gravity & Emergent Spacetime",
        "abstract": "This paper presents a matrix formulation of scalaron quantum gravity, showing how spacetime emerges from the underlying resonant field structure. We derive the Einstein field equations as an effective description of the scalaron condensate dynamics.",
        "note": "This is a placeholder PDF. The actual paper content is available in DOCX format."
    }
}

def create_placeholder_pdf(filename, metadata):
    """Create a placeholder PDF with basic information."""
    pdf_path = os.path.join(PAPERS_DIR, f"{filename}.pdf")
    
    # Create PDF document
    doc = SimpleDocTemplate(pdf_path, pagesize=letter)
    story = []
    
    # Get styles
    styles = getSampleStyleSheet()
    title_style = ParagraphStyle(
        'CustomTitle',
        parent=styles['Title'],
        fontSize=24,
        spaceAfter=30,
        alignment=TA_CENTER
    )
    
    # Add content
    story.append(Paragraph(metadata['title'], title_style))
    story.append(Spacer(1, 0.5*inch))
    
    story.append(Paragraph("<b>Abstract</b>", styles['Heading2']))
    story.append(Spacer(1, 0.2*inch))
    story.append(Paragraph(metadata['abstract'], styles['Normal']))
    story.append(Spacer(1, 0.5*inch))
    
    # Add placeholder note
    note_style = ParagraphStyle(
        'Note',
        parent=styles['Normal'],
        fontSize=10,
        textColor='gray',
        alignment=TA_CENTER
    )
    story.append(Spacer(1, inch))
    story.append(Paragraph(metadata['note'], note_style))
    
    # Build PDF
    doc.build(story)
    print(f"Created placeholder PDF: {filename}.pdf")

def main():
    # Ensure directory exists
    os.makedirs(PAPERS_DIR, exist_ok=True)
    
    # Create placeholder PDFs
    for filename, metadata in papers.items():
        create_placeholder_pdf(filename, metadata)
    
    print("\nPlaceholder PDFs created successfully!")
    print("Note: These are temporary placeholders. Use the build script with LibreOffice installed to generate actual PDFs from DOCX files.")

if __name__ == "__main__":
    main()