#!/bin/bash
# C11 Kernel-Mix Test - Final orthogonal test if C10.2 fails (<20%)
# Tests Gaussian-Lorentzian blend in Mode I spiral convolution
# 9 configs total: w ∈ {0.25, 0.50, 0.75} × sigma_L ∈ {0.40, 0.50, 0.60}

set -e

TRAIN="cases/SP99-TRAIN.manifest.txt"
BASE="config/global_c10p_best.json"

if [ ! -f "$BASE" ]; then
  echo "❌ Base config not found: $BASE"
  echo "Using fallback: config/global_c10p_tail.json"
  BASE="config/global_c10p_tail.json"
fi

echo "=== C11 Kernel-Mix Test ==="
echo "Testing Gaussian-Lorentzian blend for Mode I spiral"
echo "Base config: $BASE"
echo "9 configs: w × sigma_L"
echo "IMPORTANT: This requires implementation of w_lorentz, sigma_L in sparc_rft/kernels.py"
echo "If not implemented, this script will FAIL - treat as NOT READY."
echo ""

# Check if kernel-mix is implemented
if ! grep -q "w_lorentz" sparc_rft/kernels.py 2>/dev/null; then
  echo "⚠️  WARNING: Kernel-mix not implemented in sparc_rft/kernels.py"
  echo "This is a PLACEHOLDER script for future work."
  echo ""
  echo "To implement C11:"
  echo "  1. Modify sparc_rft/kernels.py to support Gaussian-Lorentzian blend"
  echo "  2. Add config keys: mode_spiral.w_lorentz, mode_spiral.sigma_L"
  echo "  3. Re-run this script"
  echo ""
  exit 1
fi

best_pass=0
best_config=""
best_w=0
best_sL=0

for w in 0.25 0.50 0.75; do
  for sL in 0.40 0.50 0.60; do
    echo "[w=$w sigma_L=$sL] Generating config..."

    jq --arg w "$w" --arg sL "$sL" '
      .mode_spiral.w_lorentz = ($w|tonumber) |
      .mode_spiral.sigma_L = ($sL|tonumber) |
      .mode_spiral.sigma_ln_r = 0.50 |
      .mode_shelf.A_shelf = 0
    ' "$BASE" > config/c11_try.json

    echo "[w=$w sigma_L=$sL] Running solver on TRAIN..."
    python3 -m cli.rft_rc_bench --batch "$TRAIN" \
      --solver rft_geom --global-config config/c11_try.json \
      --min-radius 1.0 --max-radius 30.0 --min-points 10 --emit-curves --max-workers 0

    echo "[w=$w sigma_L=$sL] Aggregating..."
    python3 -m batch.aggregate --restrict-manifest "$TRAIN" \
      --pass-threshold 20.0 --suffix C11_TRAIN

    passes=$(jq '[.rows[] | select(.solvers.rft_geom.rms_pct<=20)] | length' reports/_summary_C11_TRAIN.json)
    total=$(jq '.solver_totals.rft_geom.n_cases' reports/_summary_C11_TRAIN.json)
    medrms=$(jq '.solver_totals.rft_geom.rms_pct_median' reports/_summary_C11_TRAIN.json)
    pct=$(python3 -c "print(f'{$passes*100.0/$total:.1f}')")

    echo "[w=$w sigma_L=$sL] Result: $passes/$total @ 20% = $pct%  |  Median RMS = $medrms%"

    if [ $passes -gt $best_pass ]; then
      best_pass=$passes
      best_config="w=$w sigma_L=$sL"
      best_w=$w
      best_sL=$sL
      cp config/c11_try.json config/c11_best.json
      echo "         ⭐ NEW BEST!"
    fi
    echo ""
  done
done

echo "=== C11 Best Config ==="
echo "Parameters: $best_config"
echo "TRAIN performance: $best_pass @ 20%"
echo "Config saved to: config/c11_best.json"
echo ""

# Run gate
echo "=== C11 Gate Decision ==="
python3 scripts/post_c10p_gate.py --summary reports/_summary_C11_TRAIN.json

echo ""
echo "If C11 also <25%, close rotation curve campaign as negative result."
