#!/bin/bash
# C10.2 micro-tune script - ONE quick iteration for YELLOW zone (20-30%)
# Usage: ./scripts/c10p_microtune.sh <option>
#   option A: soften_inner (r_turn=2.0, A_flat=0.28)
#   option B: deripple (sigma_ln_r=0.55)

set -e

if [ $# -ne 1 ]; then
  echo "Usage: $0 <option>"
  echo "  option A: soften_inner"
  echo "  option B: deripple"
  exit 1
fi

OPTION=$1
TRAIN="cases/SP99-TRAIN.manifest.txt"
BASE="config/global_c10p_best.json"

if [ ! -f "$BASE" ]; then
  echo "❌ Base config not found: $BASE"
  echo "Run grid search first to generate global_c10p_best.json"
  exit 1
fi

case $OPTION in
  A|soften_inner)
    echo "=== C10.2 Micro-tune: Soften Inner ==="
    echo "Modify: r_turn_kpc=2.0, A_flat=0.28"
    echo "Keep: A0, alpha from best config"
    echo ""

    jq '
      .mode_tail.r_turn_kpc = 2.0 |
      .mode_tail.p = 2.0 |
      .mode_flattening.A_flat = 0.28
    ' "$BASE" > config/c10p_microtune.json

    echo "Running solver on TRAIN..."
    python3 -m cli.rft_rc_bench --batch "$TRAIN" \
      --solver rft_geom --global-config config/c10p_microtune.json \
      --min-radius 1.0 --max-radius 30.0 --min-points 10 --emit-curves --max-workers 0

    echo "Aggregating..."
    python3 -m batch.aggregate --restrict-manifest "$TRAIN" \
      --pass-threshold 20.0 --suffix C10P_MICROTUNE

    echo ""
    echo "=== Micro-tune Results ==="
    python3 scripts/post_c10p_gate.py --summary reports/_summary_C10P_MICROTUNE.json

    ;;

  B|deripple)
    echo "=== C10.2 Micro-tune: De-ripple ==="
    echo "Modify: sigma_ln_r=0.55 (widen smoothing kernel)"
    echo "Keep: A0, alpha, A_flat from best config"
    echo ""

    jq '
      .mode_spiral.sigma_ln_r = 0.55
    ' "$BASE" > config/c10p_microtune.json

    echo "Running solver on TRAIN..."
    python3 -m cli.rft_rc_bench --batch "$TRAIN" \
      --solver rft_geom --global-config config/c10p_microtune.json \
      --min-radius 1.0 --max-radius 30.0 --min-points 10 --emit-curves --max-workers 0

    echo "Aggregating..."
    python3 -m batch.aggregate --restrict-manifest "$TRAIN" \
      --pass-threshold 20.0 --suffix C10P_MICROTUNE

    echo ""
    echo "=== Micro-tune Results ==="
    python3 scripts/post_c10p_gate.py --summary reports/_summary_C10P_MICROTUNE.json

    ;;

  *)
    echo "❌ Unknown option: $OPTION"
    echo "Valid options: A (soften_inner), B (deripple)"
    exit 1
    ;;
esac
