#!/usr/bin/env python3
"""
Generate the polished RFT v2 PDF used on the public site.

This script stitches together the executive summary, reader guidance,
and the two primary markdown sources:
  - FAIR_COMPARISON_RESULTS.md
  - RFT_V2.1_FINAL_REPORT.md

It then runs pandoc -> wkhtmltopdf with professional margins,
headers/footers, and custom styling.
"""

from __future__ import annotations

import datetime as _dt
import subprocess
import textwrap
from pathlib import Path

PROJECT_ROOT = Path(__file__).resolve().parents[1]
SOURCE_DIR = PROJECT_ROOT / "reports" / "rft_v2_pdf"
STYLESHEET = SOURCE_DIR / "styles.css"
COMPILED_MD = SOURCE_DIR / "rft_v2_report_compiled.md"
OUTPUT_PDF = PROJECT_ROOT / "app" / "static" / "papers" / "rft-v2-galaxy-rotations-draft.pdf"


def _read_markdown(rel_path: str) -> str:
    path = PROJECT_ROOT / rel_path
    return path.read_text(encoding="utf-8").strip()


def _page_break() -> str:
    return '\n<div class="page-break"></div>\n'


def build_markdown() -> str:
    fair = _read_markdown("FAIR_COMPARISON_RESULTS.md")
    refine = _read_markdown("RFT_V2.1_FINAL_REPORT.md")
    today = _dt.date.today().strftime("%B %d, %Y")

    chrome = textwrap.dedent(
        f"""
        <div class="pdf-header">
          <div class="row">
            <span class="title">RFT v2 Galaxy Rotation Curves</span>
            <span class="page-counter"></span>
          </div>
        </div>

        <div class="pdf-footer">
          <div class="row">
            <span>RFT Cosmology Project — November 2025</span>
            <span>Generated on {today}</span>
          </div>
        </div>
        """
    ).strip()

    chrome_block = textwrap.indent(chrome, "        ")

    exec_summary = textwrap.dedent(
        f"""
        ---
        title: "RFT v2 Galaxy Rotation Curves"
        author: "RFT Cosmology Project"
        date: "{today}"
        ---

        {chrome_block}

        # Executive Summary

        Predictive rotation-curve modeling demands fair parameter budgets.  This report freezes the
        RFT v2 acceleration-gated solver (six global parameters, zero per-galaxy tuning) and benchmarks
        it against global NFW and MOND baselines on a blind TEST cohort of 34 SPARC galaxies.

        | Model | TEST pass@20% | Parameter budget | Framing |
        |-------|---------------|------------------|---------|
        | **RFT v2** | **58.8% (20/34)** | k=0, 6 global parameters | Predictive geometry-only tail |
        | NFW$_{{\\text{{global}}}}$ | 52.9% (18/34) | k=0, 2 global | Frozen halo $(\\rho_s, r_s)$ |
        | MOND | 23.5% (8/34) | k=0, 1 global ($a_0$) | Canonical interpolation |

        <div class="summary-grid">
        <div class="summary-card">
        <strong>Primary statistic</strong><br/>
        McNemar exact (paired) vs NFW: <strong>p = 0.69</strong><br/>
        Interpretation: RFT is competitive, not yet significant.
        </div>
        <div class="summary-card">
        <strong>MOND comparison</strong><br/>
        McNemar exact vs MOND: <strong>p = 0.004</strong><br/>
        RFT wins 14 discordant galaxies vs 2 for MOND.
        </div>
        <div class="summary-card">
        <strong>Mechanism validation</strong><br/>
        Low-surface-brightness cohort: <strong>66.7% vs 0%</strong><br/>
        Acceleration gate activates where baryons are weak.
        </div>
        <div class="summary-card">
        <strong>Robustness</strong><br/>
        ±10% perturbations on all six parameters: <strong>0 degradation</strong>.<br/>
        Ablations show the tail term drives the lift (−35pp when removed).
        </div>
        </div>

        ::: {{.definition-box}}
        **k=0 Fair Comparison.** All models share the same per-galaxy budget (zero).  Global parameters
        are fitted on TRAIN, frozen, and evaluated once on blind TEST data.  This isolates predictive value
        from descriptive curve-fitting freedom (e.g., k=2 per-galaxy halos).
        :::

        ::: {{.definition-box .callout-warning}}
        **Reader note.** The oft-quoted 82.4% NFW number corresponds to k=2 descriptive fits.
        We include it in the appendix for transparency but do not compare it directly to k=0 results.
        :::
        """
    ).strip()

    roadmap = textwrap.dedent(
        """
        # Reader Roadmap

        ## How to Use This Document

        1. **Executive summary** (previous page) gives the research question, key statistics, and terminology.
        2. **Part I** contains the full fair-comparison analysis, including the head-to-head matrix,
           Wilson confidence intervals, and paired test methodology.
        3. **Part II** documents the v2.1 refinement grid search confirming that the frozen v2 parameters
           form a local optimum without further tuning.
        4. **Appendices** (within the source markdown) retain all tables, figures, and reproducibility notes
           from the underlying reports.

        ## New to RFT?

        - RFT models add a geometry-derived tail acceleration to baryonic gravity without invoking
          per-galaxy halos.  The acceleration gate activates in low-$g_b$ environments, naturally favoring
          low surface brightness galaxies.
        - A **predictive** claim means no per-galaxy tuning after the TRAIN/TEST split.  All kaizen happens
          on TRAIN; TEST is evaluated blind exactly once.
        - Statistical honesty matters: McNemar exact tests the discordant galaxies directly; Wilson intervals
          communicate the uncertainty from a 34-galaxy TEST cohort.
        """
    ).strip()

    part_one = "\n\n".join(
        [
            "# Part I – Fair Comparison Results",
            "The following pages reproduce `FAIR_COMPARISON_RESULTS.md` verbatim for full transparency.",
            fair,
        ]
    )

    part_two = "\n\n".join(
        [
            "# Part II – v2.1 Shape Refinement Study",
            "Full contents of `RFT_V2.1_FINAL_REPORT.md`, including preregistration details, grid definition, and TEST audit.",
            refine,
        ]
    )

    return (
        exec_summary
        + _page_break()
        + roadmap
        + _page_break()
        + part_one
        + _page_break()
        + part_two
        + "\n"
    )


def run_pandoc(markdown_path: Path) -> None:
    cmd = [
        "pandoc",
        str(markdown_path),
        "--pdf-engine=wkhtmltopdf",
        "--toc",
        "--toc-depth=2",
        "--css",
        str(STYLESHEET),
        "--resource-path",
        str(PROJECT_ROOT),
        "-o",
        str(OUTPUT_PDF),
    ]

    wkhtml_opts = [
        ("--margin-left", "32mm"),
        ("--margin-right", "32mm"),
        ("--margin-top", "32mm"),
        ("--margin-bottom", "28mm"),
    ]

    for opt, value in wkhtml_opts:
        cmd.append(f"--pdf-engine-opt={opt}")
        if value:
            cmd.append(f"--pdf-engine-opt={value}")

    subprocess.run(cmd, check=True)


def main() -> int:
    SOURCE_DIR.mkdir(parents=True, exist_ok=True)
    markdown = build_markdown()
    COMPILED_MD.write_text(markdown, encoding="utf-8")
    run_pandoc(COMPILED_MD)
    print(f"✅ PDF generated at {OUTPUT_PDF}")
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
