#!/usr/bin/env node

const { execSync } = require('child_process');
const fs = require('fs');
const path = require('path');

const PAPERS_DIR = path.join(__dirname, '..', 'static', 'papers', '15x');

// Check if LibreOffice is available
function checkLibreOffice() {
    try {
        execSync('which libreoffice', { stdio: 'ignore' });
        return 'libreoffice';
    } catch {
        try {
            execSync('which soffice', { stdio: 'ignore' });
            return 'soffice';
        } catch {
            console.error('Error: LibreOffice is not installed. Please install it first:');
            console.error('  sudo apt-get install libreoffice');
            process.exit(1);
        }
    }
}

// Convert DOCX to PDF
function convertDocxToPdf(docxPath, libreofficeCmd) {
    const fileName = path.basename(docxPath, '.docx');
    const pdfPath = path.join(PAPERS_DIR, `${fileName}.pdf`);
    
    console.log(`Converting: ${fileName}.docx -> ${fileName}.pdf`);
    
    try {
        execSync(`${libreofficeCmd} --headless --convert-to pdf --outdir "${PAPERS_DIR}" "${docxPath}"`, {
            stdio: 'inherit'
        });
        
        if (fs.existsSync(pdfPath)) {
            console.log(`✓ Successfully converted: ${fileName}.pdf`);
            return true;
        } else {
            console.error(`✗ Failed to convert: ${fileName}.docx`);
            return false;
        }
    } catch (error) {
        console.error(`✗ Error converting ${fileName}.docx:`, error.message);
        return false;
    }
}

// Check timestamps
function checkTimestamps() {
    const files = fs.readdirSync(PAPERS_DIR);
    const docxFiles = files.filter(f => f.endsWith('.docx'));
    let failedChecks = 0;
    
    console.log('\nChecking timestamps...');
    
    for (const docxFile of docxFiles) {
        const docxPath = path.join(PAPERS_DIR, docxFile);
        const pdfFile = docxFile.replace('.docx', '.pdf');
        const pdfPath = path.join(PAPERS_DIR, pdfFile);
        
        if (fs.existsSync(pdfPath)) {
            const docxStats = fs.statSync(docxPath);
            const pdfStats = fs.statSync(pdfPath);
            
            if (pdfStats.mtime < docxStats.mtime) {
                console.log(`✗ PDF is older than DOCX: ${docxFile.replace('.docx', '')}`);
                failedChecks++;
            } else {
                console.log(`✓ PDF is up to date: ${docxFile.replace('.docx', '')}`);
            }
        } else {
            console.log(`✗ Missing PDF for: ${docxFile}`);
            failedChecks++;
        }
    }
    
    return failedChecks;
}

// Main function
async function main() {
    // Ensure directory exists
    if (!fs.existsSync(PAPERS_DIR)) {
        fs.mkdirSync(PAPERS_DIR, { recursive: true });
    }
    
    // Check for LibreOffice
    const libreofficeCmd = checkLibreOffice();
    
    console.log(`Converting DOCX files to PDF in ${PAPERS_DIR}...`);
    
    // Get all DOCX files
    const files = fs.readdirSync(PAPERS_DIR);
    const docxFiles = files.filter(f => f.endsWith('.docx'));
    
    if (docxFiles.length === 0) {
        console.log('No DOCX files found in the directory.');
        return;
    }
    
    // Convert each file
    let successCount = 0;
    for (const docxFile of docxFiles) {
        const docxPath = path.join(PAPERS_DIR, docxFile);
        if (convertDocxToPdf(docxPath, libreofficeCmd)) {
            successCount++;
        }
    }
    
    console.log(`\nConverted ${successCount}/${docxFiles.length} files successfully.`);
    
    // Check timestamps
    const failedChecks = checkTimestamps();
    
    if (failedChecks > 0) {
        console.error(`\nError: ${failedChecks} file(s) failed timestamp check`);
        process.exit(1);
    } else {
        console.log('\nAll PDFs are up to date!');
    }
}

// Run the script
main().catch(error => {
    console.error('Build failed:', error);
    process.exit(1);
});