#!/usr/bin/env python3
"""Benchmark Pauli vs Dirac fine-structure splittings."""

from __future__ import annotations

import argparse
import csv
import os
from pathlib import Path

from rft_atomic.engine import build_system
from rft_atomic.engine.solver_single import fine_structure, reset_spin_orbit_cache


def _split(system, n):
    grid = [(n, 1, 1.5), (n, 1, 0.5)]
    return fine_structure(system, grid)[(grid[0], grid[1])]


def main() -> int:
    parser = argparse.ArgumentParser(description="Compare Pauli vs Dirac ΔFS estimates.")
    parser.add_argument(
        "--symbols",
        nargs="+",
        default=["Na", "K", "Rb", "Cs", "Fr", "Ca+", "Ba+"],
        help="Species to benchmark.",
    )
    parser.add_argument(
        "--out",
        default="results/v063_dirac_compare.csv",
        help="Output CSV path.",
    )
    args = parser.parse_args()

    rows = []
    for symbol in args.symbols:
        reset_spin_orbit_cache()
        os.environ.pop("RFT_REL_MODE", None)
        pauli = build_system(symbol)
        split_pauli = _split(pauli, pauli.n_val)

        reset_spin_orbit_cache()
        os.environ["RFT_REL_MODE"] = "dirac"
        dirac = build_system(symbol)
        split_dirac = _split(dirac, dirac.n_val)

        rows.append(
            {
                "symbol": symbol,
                "pauli_cm": abs(split_pauli) * 219474.6313705,
                "dirac_cm": abs(split_dirac) * 219474.6313705,
            }
        )

    out_path = Path(args.out)
    out_path.parent.mkdir(parents=True, exist_ok=True)
    with out_path.open("w", newline="", encoding="utf-8") as handle:
        writer = csv.DictWriter(handle, fieldnames=["symbol", "pauli_cm", "dirac_cm"])
        writer.writeheader()
        writer.writerows(rows)

    print(f"✓ Results written to {out_path}")
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
