#!/usr/bin/env python3
"""
Compare WKB vs Calogero delta refinement for single-valence species.

Outputs a CSV summarising the quantum defects and ground-state ionisation
energies under both solvers.  Uses the same infrastructure as the main bench
script but flips the RFT_DELTA_SOLVER flag.
"""

from __future__ import annotations

import argparse
import csv
import os
from pathlib import Path

from rft_atomic.engine import build_system, registry
from rft_atomic.engine.solver_single import compute_state


def _sample(symbol: str, solver: str) -> dict[str, float]:
    os.environ["RFT_DELTA_SOLVER"] = solver
    system = build_system(symbol)
    j_ground = 0.5 if system.l_val == 0 else max(system.l_val - 0.5, 0.5)
    state = compute_state(system, system.n_val, system.l_val, j_ground)
    return {
        "delta_s": system.core.delta_s,
        "delta_p": system.core.delta_p,
        "delta_d": system.core.delta_d,
        "ionization_eV": -state["energy_eV"],
    }


def main() -> int:
    parser = argparse.ArgumentParser(description="Compare WKB vs Calogero delta solvers.")
    parser.add_argument(
        "--symbols",
        nargs="+",
        default=["Li", "Na", "K", "Rb", "Cs", "Mg+", "Ca+", "Sr+", "Ba+"],
        help="Species to sample.",
    )
    parser.add_argument(
        "--element-file",
        action="append",
        metavar="PATH",
        help="Optional YAML/JSON registry pack to load.",
    )
    parser.add_argument(
        "--out",
        default="results/v063_calo_compare.csv",
        help="Output CSV path.",
    )
    args = parser.parse_args()

    if args.element_file:
        for pack in args.element_file:
            registry.load_external_registry(pack)

    rows = []
    for symbol in args.symbols:
        wkb = _sample(symbol, "wkb")
        calo = _sample(symbol, "calogero")
        rows.append(
            {
                "symbol": symbol,
                "delta_s_wkb": wkb["delta_s"],
                "delta_s_calo": calo["delta_s"],
                "delta_p_wkb": wkb["delta_p"],
                "delta_p_calo": calo["delta_p"],
                "delta_d_wkb": wkb["delta_d"],
                "delta_d_calo": calo["delta_d"],
                "ionization_wkb_eV": wkb["ionization_eV"],
                "ionization_calo_eV": calo["ionization_eV"],
            }
        )

    out_path = Path(args.out)
    out_path.parent.mkdir(parents=True, exist_ok=True)
    with out_path.open("w", newline="", encoding="utf-8") as handle:
        writer = csv.DictWriter(
            handle,
            fieldnames=[
                "symbol",
                "delta_s_wkb",
                "delta_s_calo",
                "delta_p_wkb",
                "delta_p_calo",
                "delta_d_wkb",
                "delta_d_calo",
                "ionization_wkb_eV",
                "ionization_calo_eV",
            ],
        )
        writer.writeheader()
        writer.writerows(rows)

    print(f"✓ Results written to {out_path}")
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
