#!/usr/bin/env python3
"""Batch convert SPARC rotmod files to GalaxyCase JSON format."""

import sys
from pathlib import Path

# Add project root to path
PROJECT_ROOT = Path(__file__).resolve().parents[1]
if str(PROJECT_ROOT) not in sys.path:
    sys.path.insert(0, str(PROJECT_ROOT))

from ingest.sparc_loader import load_rotmod_case
from sparc_rft.case import dump_case


def main():
    import argparse
    parser = argparse.ArgumentParser(description="Batch convert SPARC rotmod to JSON")
    parser.add_argument("--manifest", required=True, help="Manifest file with rotmod paths")
    parser.add_argument("--outdir", required=True, help="Output directory for JSON files")
    args = parser.parse_args()

    manifest_path = Path(args.manifest)
    out_dir = Path(args.outdir)
    out_dir.mkdir(parents=True, exist_ok=True)

    with manifest_path.open() as f:
        rotmod_paths = [Path(line.strip()) for line in f if line.strip()]

    print(f"Converting {len(rotmod_paths)} rotmod files to JSON...")

    success_count = 0
    error_count = 0

    for rotmod_path in rotmod_paths:
        try:
            # Load rotmod and convert to GalaxyCase
            case = load_rotmod_case(rotmod_path)

            # Output filename: galaxy_name.json
            galaxy_slug = rotmod_path.stem.replace("_rotmod", "")
            out_path = out_dir / f"{galaxy_slug}.json"

            # Write JSON
            dump_case(case, out_path)
            success_count += 1

        except Exception as e:
            print(f"ERROR: {rotmod_path.name}: {e}")
            error_count += 1

    print(f"\nCompleted: {success_count} successful, {error_count} errors")
    if error_count > 0:
        print(f"⚠️  {error_count} files failed to convert")
    else:
        print(f"✅ All {success_count} files converted successfully")


if __name__ == "__main__":
    main()
