#!/usr/bin/env python3
"""
Auto-fill SPARC120_RESULTS_TEMPLATE.md from reports/_summary_SP120.json
"""

import json
import hashlib
from pathlib import Path
from datetime import datetime


def compute_sha256(path):
    """Compute SHA256 hash of a file."""
    sha = hashlib.sha256()
    with open(path, 'rb') as f:
        while chunk := f.read(8192):
            sha.update(chunk)
    return sha.hexdigest()


def main():
    # Load summary
    summary_path = Path("reports/_summary_SP120.json")
    if not summary_path.exists():
        print(f"❌ Summary file not found: {summary_path}")
        print("   Run the pipeline first: bash scripts/after_rft.sh")
        return 1

    with summary_path.open() as f:
        summary = json.load(f)

    # Extract data
    meta = summary.get("meta", {})
    solver_totals = summary.get("solver_totals", {})
    wins = summary.get("wins", {})
    rows = summary.get("rows", [])

    rft = solver_totals.get("rft_geom", {})
    mond = solver_totals.get("mond", {})
    nfw = solver_totals.get("nfw_fit", {})

    # Compute LSB/HSB stats
    lsb_rows = [r for r in rows if "LSB" in r.get("tags", [])]
    hsb_rows = [r for r in rows if "HSB" in r.get("tags", [])]

    lsb_rft_pass = sum(1 for r in lsb_rows if r.get("solvers", {}).get("rft_geom", {}).get("pass", False))
    lsb_mond_pass = sum(1 for r in lsb_rows if r.get("solvers", {}).get("mond", {}).get("pass", False))
    hsb_rft_pass = sum(1 for r in hsb_rows if r.get("solvers", {}).get("rft_geom", {}).get("pass", False))
    hsb_mond_pass = sum(1 for r in hsb_rows if r.get("solvers", {}).get("mond", {}).get("pass", False))

    lsb_rft_rms = [r.get("solvers", {}).get("rft_geom", {}).get("rms_pct") for r in lsb_rows]
    lsb_rft_rms = [x for x in lsb_rft_rms if x is not None]
    hsb_rft_rms = [r.get("solvers", {}).get("rft_geom", {}).get("rms_pct") for r in hsb_rows]
    hsb_rft_rms = [x for x in hsb_rft_rms if x is not None]

    # Compute outliers
    all_rms = [r.get("solvers", {}).get("rft_geom", {}).get("rms_pct") for r in rows]
    all_rms = [x for x in all_rms if x is not None]
    outliers = [r for r in rows if r.get("solvers", {}).get("rft_geom", {}).get("rms_pct", 0) > 25.0]

    # Pairwise stats
    rft_vs_mond = wins.get("rft_geom>mond", {})
    rft_vs_nfw = wins.get("rft_geom>nfw_fit", {})
    nfw_vs_mond = wins.get("nfw_fit>mond", {})

    # Manifest SHA
    manifest_path = Path("cases/SPARC-120.manifest.txt")
    manifest_sha = compute_sha256(manifest_path) if manifest_path.exists() else "unknown"

    # Build output
    output = f"""# SPARC-120 Validation Results

**Auto-filled from**: `reports/_summary_SP120.json`
**Date**: {meta.get('timestamp_utc', 'unknown')}
**Config SHA**: {meta.get('global_config_sha256', 'unknown')[:16]}...
**Cohort**: 120 galaxies (balanced LSB/HSB, min 10 points, seed 42)

---

## Executive Summary

### Primary Result: RFT Pass Rate
- **Pass Rate**: {rft.get('pass_rate', 0)*100:.1f}% ({rft.get('pass_count', 0)}/{rft.get('n_cases', 0)})
- **95% Wilson CI**: [{rft.get('wilson_95', [0, 0])[0]*100:.1f}%, {rft.get('wilson_95', [0, 0])[1]*100:.1f}%]
- **Median RMS%**: {rft.get('rms_pct_median', 0):.1f}%

**Interpretation**:
"""

    # Interpretation logic
    pass_rate = rft.get('pass_rate', 0)
    if pass_rate >= 0.80:
        output += "✅ **STRONG SUPPORT** for no-retune claim. RFT achieves ≥80% pass rate with a single frozen configuration across 120 diverse galaxies.\n"
    elif pass_rate >= 0.60:
        output += "✅ **MODERATE SUPPORT** for no-retune claim. RFT achieves ≥60% pass rate, demonstrating broad applicability without per-galaxy tuning.\n"
    else:
        output += "⚠️  **BELOW TARGET**. RFT pass rate <60% suggests need for model refinement or parameter adjustment.\n"

    output += f"""
---

## Comparative Performance

### Pass Rates by Solver

| Solver | Pass Rate | 95% CI | Median RMS% | N |
|--------|-----------|--------|-------------|---|
| **RFT (Geometric)** | {rft.get('pass_rate', 0)*100:.1f}% | [{rft.get('wilson_95', [0, 0])[0]*100:.1f}%, {rft.get('wilson_95', [0, 0])[1]*100:.1f}%] | {rft.get('rms_pct_median', 0):.1f}% | {rft.get('n_cases', 0)} |
| NFW (Dark Matter) | {nfw.get('pass_rate', 0)*100:.1f}% | [{nfw.get('wilson_95', [0, 0])[0]*100:.1f}%, {nfw.get('wilson_95', [0, 0])[1]*100:.1f}%] | {nfw.get('rms_pct_median', 0):.1f}% | {nfw.get('n_cases', 0)} |
| MOND | {mond.get('pass_rate', 0)*100:.1f}% | [{mond.get('wilson_95', [0, 0])[0]*100:.1f}%, {mond.get('wilson_95', [0, 0])[1]*100:.1f}%] | {mond.get('rms_pct_median', 0):.1f}% | {mond.get('n_cases', 0)} |

### Head-to-Head Wins (Lower RMS% Wins)

| Comparison | RFT Wins | Total | Win Rate |
|------------|----------|-------|----------|
| RFT vs MOND | {rft_vs_mond.get('wins', 0)} | {rft_vs_mond.get('of', 0)} | {rft_vs_mond.get('wins', 0)/max(rft_vs_mond.get('of', 1), 1)*100:.1f}% |
| RFT vs NFW | {rft_vs_nfw.get('wins', 0)} | {rft_vs_nfw.get('of', 0)} | {rft_vs_nfw.get('wins', 0)/max(rft_vs_nfw.get('of', 1), 1)*100:.1f}% |
| NFW vs MOND | {nfw_vs_mond.get('wins', 0)} | {nfw_vs_mond.get('of', 0)} | {nfw_vs_mond.get('wins', 0)/max(nfw_vs_mond.get('of', 1), 1)*100:.1f}% |

**Interpretation**:
"""

    rft_vs_mond_pct = rft_vs_mond.get('wins', 0) / max(rft_vs_mond.get('of', 1), 1)
    rft_vs_nfw_pct = rft_vs_nfw.get('wins', 0) / max(rft_vs_nfw.get('of', 1), 1)

    if rft_vs_mond_pct > 0.90 and rft_vs_nfw_pct > 0.70:
        output += "✅ **COMPELLING EVIDENCE**: RFT wins >90% vs MOND and >70% vs NFW, demonstrating superiority over both alternative models.\n"
    elif rft_vs_mond_pct > 0.60:
        output += "✅ **MODERATE ADVANTAGE**: RFT outperforms MOND consistently, competitive with NFW dark matter fits.\n"
    else:
        output += "⚠️  **MIXED RESULTS**: RFT does not consistently outperform baselines.\n"

    output += f"""
---

## Subclass Analysis: LSB vs HSB

### Low Surface Brightness (LSB) Galaxies
- **N**: {len(lsb_rows)} galaxies
- **RFT Pass Rate**: {lsb_rft_pass/max(len(lsb_rows), 1)*100:.1f}%
- **RFT Median RMS%**: {sorted(lsb_rft_rms)[len(lsb_rft_rms)//2] if lsb_rft_rms else 0:.1f}%
- **MOND Pass Rate**: {lsb_mond_pass/max(len(lsb_rows), 1)*100:.1f}% (baseline)

"""

    if lsb_rft_pass / max(len(lsb_rows), 1) > 0.60:
        output += "✅ **SIGNIFICANT**: RFT holds up on LSB galaxies, traditionally challenging for alternative gravity models.\n"
    else:
        output += "⚠️  **CONCERN**: LSB performance below expectations.\n"

    output += f"""
### High Surface Brightness (HSB) Galaxies
- **N**: {len(hsb_rows)} galaxies
- **RFT Pass Rate**: {hsb_rft_pass/max(len(hsb_rows), 1)*100:.1f}%
- **RFT Median RMS%**: {sorted(hsb_rft_rms)[len(hsb_rft_rms)//2] if hsb_rft_rms else 0:.1f}%
- **MOND Pass Rate**: {hsb_mond_pass/max(len(hsb_rows), 1)*100:.1f}% (baseline)

---

## RMS% Distribution

- **5th percentile**: {sorted(all_rms)[len(all_rms)//20] if all_rms else 0:.1f}%
- **Median**: {sorted(all_rms)[len(all_rms)//2] if all_rms else 0:.1f}%
- **95th percentile**: {sorted(all_rms)[int(len(all_rms)*0.95)] if all_rms else 0:.1f}%

**Outliers** (RMS% > 25%):
- **Count**: {len(outliers)} galaxies
- **Fraction**: {len(outliers)/max(len(rows), 1)*100:.1f}%

"""

    if outliers:
        outlier_names = ", ".join([r.get("name", "unknown") for r in outliers[:5]])
        output += f"Notable outliers: {outlier_names}{'...' if len(outliers) > 5 else ''}\n"

    output += f"""
---

## Key Findings

1. **No-Retune Validation**:
   - RFT achieves {rft.get('pass_rate', 0)*100:.1f}% pass rate with **zero per-galaxy parameter tuning**
   - Configuration frozen at SHA `{meta.get('global_config_sha256', 'unknown')[:8]}`
   - Pass threshold: 20% RMS% (industry-standard)

2. **Baseline Comparisons**:
   - RFT outperforms MOND in {rft_vs_mond_pct*100:.1f}% of pairwise comparisons
   - RFT outperforms NFW dark matter in {rft_vs_nfw_pct*100:.1f}% of comparisons
   - Clear performance hierarchy: {"RFT > NFW > MOND" if rft_vs_mond_pct > 0.9 and rft_vs_nfw_pct > 0.7 else "See matrix for details"}

3. **LSB Galaxy Performance**:
   - {lsb_rft_pass/max(len(lsb_rows), 1)*100:.1f}% pass rate on LSB subset
   - Critical test for alternative gravity (traditionally MOND's strength)
   - {"**RFT exceeds MOND on LSBs**" if lsb_rft_pass > lsb_mond_pass else "MOND competitive on LSBs"}

4. **Publication Figures**:
   - Pass rate bars with Wilson CIs → `reports/figs/fig_passrate_bar.png`
   - Violin plots (LSB/HSB split) → `reports/figs/fig_violin_rms_lsb_hsb.png`
   - Head-to-head matrix → `reports/figs/fig_headtohead_matrix.png`
   - Scatter (RMS% vs r_max) → `reports/figs/fig_scatter_rms_vs_rmax.png`

---

## Reproducibility

### Configuration Lock
- **Global Config**: `config/global.json` (SHA: `{meta.get('global_config_sha256', 'unknown')[:8]}`)
- **SPARC-120 Manifest**: `cases/SPARC-120.manifest.txt` (SHA: `{manifest_sha[:8]}`)
- **Selection Seed**: 42 (deterministic)
- **Balance Tags**: LSB, HSB (median disk-to-total ratio)

### Validation Command
```bash
# Reproduce SPARC-120 selection
python3 -m ingest.select_cohort \\
  --summary reports/_summary.json \\
  --n 120 --balance LSB,HSB --seed 42 --min-points 10 --min-incl 30.0 \\
  --write cases/SPARC-120.manifest.txt

# Run all 3 solvers
bash scripts/after_rft.sh
```

---

## Next Steps: Blind Split (Ticket C8)

### Pre-Registration Protocol
1. **Split SPARC-120** into SP80-train / SP40-test (balanced, seed 42)
2. **Freeze configuration** before analyzing test set
3. **Document predictions** (expected pass rate, RMS% distribution)
4. **Unblind test set** only after pre-registration complete
5. **Report results** with full provenance

**Goal**: Lock the no-retune claim with blinded, pre-registered validation.

---

**Generated**: {datetime.now().strftime('%Y-%m-%d %H:%M:%S UTC')}
"""

    # Write output
    output_path = Path("SPARC120_RESULTS.md")
    with output_path.open("w") as f:
        f.write(output)

    print(f"✅ Results auto-filled successfully!")
    print(f"   Output: {output_path}")
    print("")
    print("Key metrics:")
    print(f"  RFT Pass Rate: {rft.get('pass_rate', 0)*100:.1f}% ({rft.get('pass_count', 0)}/{rft.get('n_cases', 0)})")
    print(f"  RFT vs MOND:   {rft_vs_mond_pct*100:.1f}% wins")
    print(f"  RFT vs NFW:    {rft_vs_nfw_pct*100:.1f}% wins")
    print(f"  LSB Pass Rate: {lsb_rft_pass/max(len(lsb_rows), 1)*100:.1f}%")
    print(f"  HSB Pass Rate: {hsb_rft_pass/max(len(hsb_rows), 1)*100:.1f}%")
    print("")
    return 0


if __name__ == "__main__":
    exit(main())
