#!/bin/bash
# Analyze worst 10 galaxies by RMS to diagnose systematic failures
# Usage: ./scripts/analyze_worst10.sh reports/_summary_C9_A.json

SUMMARY="${1:-reports/_summary_C9_A.json}"
CONFIG="${2:-config/global_c9.json}"

if [ ! -f "$SUMMARY" ]; then
    echo "ERROR: Summary not found: $SUMMARY"
    exit 1
fi

echo "=== Top 10 Worst Galaxies by RMS ==="
echo ""

# Extract worst 10 - handle both single-solver and multi-solver summaries
if jq -e '.solvers' "$SUMMARY" > /dev/null 2>&1; then
    # Multi-solver summary
    jq -r '.rows[] | select(.solvers.rft_geom) | [.name, .solvers.rft_geom.rms_pct] | @tsv' "$SUMMARY" \
      | sort -k2 -nr | head -10 > worst10.txt
else
    # Single-solver summary
    jq -r '.rows[] | [.name, .rms_pct] | @tsv' "$SUMMARY" \
      | sort -k2 -nr | head -10 > worst10.txt
fi

if [ ! -s worst10.txt ]; then
    echo "No galaxies found in summary. Exiting."
    rm -f worst10.txt
    exit 0
fi

echo "Rank | Galaxy | RMS%"
echo "-----|--------|-----"
cat worst10.txt | nl -w1 -s' | ' | awk '{print $1" | "$2" | "$3"%"}'

echo ""
echo "Generating detailed residual analysis..."
echo ""

# Run solver with full diagnostics on each
mkdir -p reports/worst10_analysis

while read -r galaxy rms; do
    galaxy_file=$(echo "$galaxy" | tr ' ' '_')
    echo "Processing $galaxy (RMS: $rms%)..."

    # Run solver with --emit-curves to generate diagnostic plots
    python3 -m cli.rft_rc_bench \
        --case "cases/sparc_all/${galaxy_file}.json" \
        --solver rft_geom \
        --global-config "$CONFIG" \
        --min-radius 1.0 \
        --max-radius 30.0 \
        --min-points 10 \
        --emit-curves 2>/dev/null

    # Copy outputs to analysis directory
    if [ -d "reports/${galaxy_file}/rft_geom" ]; then
        mkdir -p "reports/worst10_analysis/$galaxy_file"
        cp -r "reports/${galaxy_file}/rft_geom"/* "reports/worst10_analysis/$galaxy_file/"
    fi

done < worst10.txt

echo ""
echo "✅ Analysis complete: reports/worst10_analysis/"
echo ""
echo "Check for patterns:"
echo "  - Outer over-prediction → reduce A_shelf or increase p"
echo "  - Inner over-prediction → reduce A_core"
echo "  - Oscillations → increase sigma_ln_r (smooth spiral mode)"
echo "  - Systematic LSB under-prediction → increase beta2 (gas sensitivity)"
echo ""
echo "View plots: ls -lh reports/worst10_analysis/*/velocities.png"
