#!/bin/bash
set -euo pipefail

echo "=== SPARC-120 Auto-Continuation Script ==="
echo "Waiting for RFT batch to complete..."

# Wait until the rft batch is no longer running
while pgrep -f "cli.rft_rc_bench.*--batch.*rft_geom" >/dev/null 2>&1; do
    sleep 10
    echo -n "."
done

echo ""
echo "✅ RFT batch completed!"
echo ""

# 1) Aggregate all-candidate RFT results
echo "📊 Step 1/5: Aggregating all-candidate RFT results..."
python3 -m batch.aggregate

# 2) Freeze SPARC-120 (balanced, seeded)
echo "📋 Step 2/5: Selecting SPARC-120 cohort (balanced LSB/HSB, seed 42)..."
python3 -m ingest.select_cohort \
  --summary reports/_summary.json \
  --n 120 --balance LSB,HSB --seed 42 \
  --min-points 10 --min-incl 30.0 \
  --write cases/SPARC-120.manifest.txt

# 3) Run all solvers on SPARC-120
echo "🔬 Step 3/5: Running all 3 solvers on SPARC-120..."
for solver in rft_geom mond nfw_fit; do
  echo "  Running $solver..."
  python3 -m cli.rft_rc_bench --batch cases/SPARC-120.manifest.txt \
    --solver "$solver" \
    --global-config config/global.json \
    --nfw-bounds "rho_s=1e6,1e9;r_s=0.1,100" \
    --min-radius 1.0 --max-radius 30.0 --min-points 10 \
    --emit-curves --max-workers 0
done

# 4) Aggregate comparative summary
echo "📈 Step 4/5: Aggregating comparative statistics..."
python3 -m batch.aggregate --suffix SP120

# 5) Export figures
echo "📸 Step 5/5: Exporting publication figures..."
python3 -m scripts.export_figs --summary reports/_summary_SP120.json --out reports/figs

# 6) Auto-fill results document
echo "📝 Step 6/6: Generating results document..."
python3 -m scripts.fill_results \
  --summary reports/_summary_SP120.json \
  --template SPARC120_RESULTS_TEMPLATE.md \
  --out SPARC120_RESULTS.md

echo ""
echo "✅ SPARC-120 PIPELINE COMPLETE!"
echo ""
echo "Results available at:"
echo "  - Summary JSON:    reports/_summary_SP120.json"
echo "  - Summary CSV:     reports/_summary_SP120.csv"
echo "  - Figures:         reports/figs/*.png"
echo "  - Results Doc:     SPARC120_RESULTS.md"
echo ""
echo "View results:"
echo "  cat SPARC120_RESULTS.md   # Full results document"
echo "  make sp120-report         # Quick terminal summary"
echo "  make publish              # Dashboard (localhost:8080)"
