#!/usr/bin/env python3
"""
Add the remaining sections to the RFT Volume 2 HTML document
"""

# Remaining sections data
sections_data = [
    {
        "id": "sec-hbar",
        "title": "S6.1 ℏ from a CP¹ Berry Loop",
        "content": """<p>Planck's constant arises as the quantised Berry phase acquired by the scalaron when it traces a canonical loop on CP¹. The loop area is fixed by the lattice geometry; imposing single-valuedness forces the phase to be an integer multiple of 2π, giving a natural unit of action. Because CP¹ has constant curvature, the resulting ℏ is independent of mesh refinement or gauge choice. The residual errors shown in the convergence plot fall well below the 0.5 % tolerance and do not grow with resolution. This topological lock is why later sensitivity tables mark ℏ as essentially immune to drift: unless the curvature or the loop topology changes, the action quantum cannot move.</p>

<h3>Why it matters</h3>
<ul>
  <li>Explains the universality and scale independence of the action quantum.</li>
  <li>Provides a reproducible geometric test that connects loop area to ℏ.</li>
  <li>Anchors the "locked" entries in the global sensitivity matrix.</li>
</ul>

<h3>Common misconceptions</h3>
<ul>
  <li>"ℏ is assumed." No—here it is derived from a Berry phase on CP¹.</li>
  <li>"Gauge choices can set ℏ." Gauge changes leave the loop phase invariant.</li>
  <li>"Refinement alters ℏ." Topology fixes the value; refinement only tightens numerics.</li>
</ul>

<h3>How to falsify</h3>
<ul>
  <li>Show the loop phase varies with discretisation or gauge.</li>
  <li>Produce an alternative CP¹ loop that yields a different ℏ without breaking topology.</li>
  <li>Demonstrate experimental evidence for scale-dependent ℏ.</li>
</ul>

<p><strong>Equation of record:</strong> ℏ ∝ ∮<sub>CP¹</sub> A · dl; single-valuedness quantises the phase.</p>

<p><strong>Glossary addendum:</strong> Berry phase; CP¹; holonomy; curvature; gauge; single-valuedness.</p>

<h3>How this section ties forward/back</h3>
<ul>
  <li>Supplies the "locked" entry used in S16.1's sensitivity table.</li>
  <li>Appears in the constants ledger consumed by every cosmological calculation.</li>
  <li>Serves as a benchmark for the no-retune rule enforced in S21.1.</li>
</ul>""",
        "figure_id": "fig-berryloop",
        "figure_src": "results/figs/h_berryloop.png",
        "figure_caption": "Figure: Berry-loop convergence and tolerance band",
        "notebook_id": "h_cp1_berry"
    },
    {
        "id": "sec-c",
        "title": "S7.1 c from Coupled-Mode Dispersion",
        "content": """<p>The speed of light emerges as the group-velocity plateau of the coupled scalaron–twistor mode. Plotting the dispersion reveals that as k → 0, the derivative dω/dk flattens to a constant. That constant is identified with c because it governs the fastest causal signal carriers in the lattice. The plateau height depends only on the coupling geometry, not on a tunable parameter, and numerical refinements confirm the plateau is stable to within the 0.1 % tolerance. The result is therefore a geometric prediction, not a definition: the same plateau must appear in any section that references light-speed propagation.</p>

<h3>Why it matters</h3>
<ul>
  <li>Provides a measurable plateau rather than assuming c by fiat.</li>
  <li>Demonstrates mesh-refinement independence for the small-k slope.</li>
  <li>Connects everyday causal speed to a specific dispersion mode.</li>
</ul>

<h3>Common misconceptions</h3>
<ul>
  <li>"c is defined via units." Here it is extracted from the dispersion plateau.</li>
  <li>"Plateaus are numerical accidents." Refinement shows the plateau persists.</li>
  <li>"You can tune c elsewhere." Any retune would break other coupled-mode checks.</li>
</ul>

<h3>How to falsify</h3>
<ul>
  <li>Show the plateau height shifts with grid density or domain size.</li>
  <li>Require ad-hoc coupling tweaks to keep c within observational bounds.</li>
  <li>Produce time-of-flight data inconsistent with the predicted plateau value.</li>
</ul>

<p><strong>Equation of record:</strong> c = lim<sub>k→0</sub> dω/dk for the physical dispersion branch.</p>

<p><strong>Glossary addendum:</strong> dispersion; group velocity; branch; small-k limit; plateau; coupled mode.</p>

<h3>How this section ties forward/back</h3>
<ul>
  <li>Locks the light-speed value used in S15.1's BAO integration.</li>
  <li>Contributes to the constants ledger referenced in S21.1.</li>
  <li>Provides a sensitivity benchmark for the "nearly locked" category in S16.1.</li>
</ul>""",
        "figure_id": "fig-c-plateau",
        "figure_src": "results/figs/c_plateau.png",
        "figure_caption": "Figure: Group-velocity plateau of the physical branch",
        "notebook_id": "c_coupled_modes"
    }
]

# Read the current file
file_path = "/home/rftuser/omega_document/admin_portal/static/html/standalone/rft-volume2-rc12.html"

with open(file_path, 'r', encoding='utf-8') as f:
    content = f.read()

# Find the insertion point (before </article>)
insertion_point = content.find('      </article>')

if insertion_point == -1:
    print("ERROR: Could not find insertion point")
    exit(1)

# Generate the sections HTML
sections_html = ""
for section in sections_data:
    section_html = f"""
<h2 id="{section['id']}">{section['title']}</h2>

{section['content']}

<figure id="{section['figure_id']}">
  <img src="{section['figure_src']}" alt="{section['figure_caption'].replace('Figure: ', '')}" />
  <figcaption>{section['figure_caption']}</figcaption>
</figure>

<p>
  <a class="nb-dl" data-nb="{section['notebook_id']}" href="results/nb/{section['notebook_id']}.zip" download="{section['notebook_id']}.zip">📦 Reproduce this figure</a>
</p>

"""
    sections_html += section_html

# Insert the sections
new_content = content[:insertion_point] + sections_html + content[insertion_point:]

# Write back to file
with open(file_path, 'w', encoding='utf-8') as f:
    f.write(new_content)

print(f"✅ Added {len(sections_data)} sections to the HTML document")
print(f"File size: {len(new_content) / 1024 / 1024:.2f} MB")